using System;
using System.Data;
using System.Reflection;
using gov.va.med.vbecs.ExceptionManagement;
using gov.va.med.vbecs.DAL.HL7AL;
using System.Collections;
using TABLE = gov.va.med.vbecs.Common.VbecsTables;
using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;

namespace gov.va.med.vbecs.BOL
{

	#region Header

		///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
		///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
		///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/9/2003</CreationDate>
		///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
		///<summary></summary>

		#endregion

	public class OrderedTest : PatientOrder, IRecordStatusCode
	{
		private Guid _orderedTestGUID;
		private Guid _relatedTestGUID;
		private Guid _relatedComponentGUID;
		private DataTable _specimenTests;
		private DataTable _bloodUnitTests;
		private string _orderableTest;
		private char _taskStatusCode;
		private string _taskStatusText;
		private string _cprsOrderNumber;
		private DateTime _requiredResultsDate;
		private int _orderableTestID;
		private int _reportableTestID;
		private string _orderUrgencyCode;
		private string _orderUrgencyText;
		private Common.OrderStatus _orderStatusCode;
		private string _orderStatusText;
		private BOL.Specimen _specimen;
		private string _specimenStatusText;
		private string _VBECSOrderNumber;
		private string _labOrderNumber;
		private string _labTestID;
		private string _specimenAccessionNumber;
		private byte[] _rowVersion;
		private Common.RecordStatusCode _recordStatusCode;

		#region constructors

		

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/15/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3231"> 
		///		<ExpectedInput>DataRow</ExpectedInput>
		///		<ExpectedOutput>OrderedTest object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3232"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Constructor
		/// </summary>
		/// <param name="dtRow"></param>
		public OrderedTest(DataRow dtRow) : base(dtRow)
		{
			LoadFromDataRow(dtRow);
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/15/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2671"> 
		///		<ExpectedInput>Guids</ExpectedInput>
		///		<ExpectedOutput>OrderedTest object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2673"> 
		///		<ExpectedInput>Invalid Guids</ExpectedInput>
		///		<ExpectedOutput>NoNullAllowedException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Ctor
		/// </summary>
		/// <param name="orderedTestGuid"></param>
		/// <param name="patientOrderGuid"></param>
		public OrderedTest(Guid orderedTestGuid, Guid patientOrderGuid) : base(patientOrderGuid)
		{
			DataRow dr = DAL.OrderedTest.GetOrderedTest(orderedTestGuid).Rows[0];
			LoadFromDataRow(dr);
		}

		#endregion


		#region properties

		///<Developers>
		///	<Developer></Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/11/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7512"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7513"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// The order could have been cancelled in CPRS before after the screen brought up, but
		/// prior to the save.  This is a check to ensure the order is still pending.
		/// </summary>
		public bool StillActive
		{
			get
			{
				if (this.OrderedTestGuid != Guid.Empty)
				{
					Common.OrderStatus status = Common.Utility.GetOrderStatusCodeFromString(DAL.OrderedTest.GetOrderedTest(this.OrderedTestGuid).Rows[0][TABLE.OrderedTest.OrderStatusCode].ToString());
					return status != Common.OrderStatus.Cancelled;
				}
				else
				{
					return false;
				}
			}
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/15/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2675"> 
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2679"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Specimen accession number from VistA
		/// </summary>
		public string SpecimenAccessionNumber
		{
			get
			{
				return this._specimenAccessionNumber;
			}
			set
			{
				this._specimenAccessionNumber = value;
			}
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/25/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2684"> 
		///		<ExpectedInput>Guid</ExpectedInput>
		///		<ExpectedOutput>Guid</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="2680"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// OrderedTestGuid
		/// </summary>
		public Guid OrderedTestGuid
		{
			get
			{
				return this._orderedTestGUID;
			}
			set
			{
				this._orderedTestGUID = value;
			}
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/18/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3233"> 
		///		<ExpectedInput>GUID</ExpectedInput>
		///		<ExpectedOutput>GUID</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="2681"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// RelatedTestGuid
		/// </summary>
		public Guid RelatedTestGuid
		{
			get
			{
				return this._relatedTestGUID;
			}
			set
			{
				this._relatedTestGUID = value;
			}
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/18/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2691"> 
		///		<ExpectedInput>Common.RecordStatusCode</ExpectedInput>
		///		<ExpectedOutput>Common.RecordStatusCode</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="2693"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// The status of the record
		/// </summary>
		public Common.RecordStatusCode RecordStatusCode
		{
			get
			{
				return _recordStatusCode;
			}
			set
			{
				_recordStatusCode = value;
			}
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/18/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2700"> 
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="2702"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Lab Test ID (from VistA)
		/// </summary>
		public string LabTestId
		{
			get
			{
				return _labTestID;
			}
			set
			{
				_labTestID = value;
			}
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/18/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3235"> 
		///		<ExpectedInput>GUID</ExpectedInput>
		///		<ExpectedOutput>GUID</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="2707"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// RelatedComponentGuid
		/// </summary>
		public Guid RelatedComponentGuid
		{
			get
			{
				return this._relatedComponentGUID;
			}
			set
			{
				this._relatedComponentGUID = value;
			}
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/25/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2685"> 
		///		<ExpectedInput>datatable</ExpectedInput>
		///		<ExpectedOutput>datatable</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="2708"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// SpecimenTests
		/// </summary>
		public DataTable SpecimenTests
		{
			get
			{
				return this._specimenTests;
			}
			set
			{
				this._specimenTests = value;
			}
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/25/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2686"> 
		///		<ExpectedInput>datatable</ExpectedInput>
		///		<ExpectedOutput>datatable</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="2709"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// BloodUnitTests
		/// </summary>
		public DataTable BloodUnitTests
		{
			get
			{
				return this._bloodUnitTests;
			}
			set
			{
				this._bloodUnitTests = value;
			}
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/25/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2687"> 
		///		<ExpectedInput>A string</ExpectedInput>
		///		<ExpectedOutput>A string</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="2710"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// OrderableTest
		/// </summary>
		public string OrderableTest
		{
			get
			{
				return this._orderableTest;
			}
			set
			{
				this._orderableTest = value;
			}
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/25/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2688"> 
		///		<ExpectedInput>A string</ExpectedInput>
		///		<ExpectedOutput>A string</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="2712"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// TaskStatusCode
		/// </summary>
		public char TaskStatusCode
		{
			get
			{
				return this._taskStatusCode;
			}
			set
			{
				this._taskStatusCode = value;
			}
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/18/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3237"> 
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="2713"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// TaskStatusText
		/// </summary>
		public string TaskStatusText
		{
			get
			{
				return this._taskStatusText;
			}
			set
			{
				this._taskStatusText = value;
			}
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/25/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2689"> 
		///		<ExpectedInput>datetime</ExpectedInput>
		///		<ExpectedOutput>datetime</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="2714"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// RequiredResultsDate
		/// </summary>
		public DateTime RequiredResultsDate
		{
			get
			{
				return this._requiredResultsDate;
			}
			set
			{
				this._requiredResultsDate = value;
			}
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/18/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3239"> 
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="2717"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// CPRSOrderNumber
		/// </summary>
		public string CPRSOrderNumber
		{
			get
			{
				return this._cprsOrderNumber;
			}
			set
			{
				this._cprsOrderNumber = value;
			}
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/18/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3241"> 
		///		<ExpectedInput>int</ExpectedInput>
		///		<ExpectedOutput>int</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="2728"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// OrderableTestId
		/// </summary>
		public int OrderableTestId
		{
			get
			{
				return this._orderableTestID;
			}
			set
			{
				this._orderableTestID = value;
			}
		}


		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/18/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2729"> 
		///		<ExpectedInput>int</ExpectedInput>
		///		<ExpectedOutput>int</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="2730"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// ReportableTestId
		/// </summary>
		public int ReportableTestId
		{
			get
			{
				return this._reportableTestID;
			}
			set
			{
				this._reportableTestID = value;
			}
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/18/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3243"> 
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="2731"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// OrderUrgencyCode
		/// </summary>
		public string OrderUrgencyCode
		{
			get
			{
				return this._orderUrgencyCode;
			}
			set
			{
				this._orderUrgencyCode = value;
			}
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/18/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3245"> 
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="2738"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// OrderUrgencyText
		/// </summary>
		public string OrderUrgencyText
		{
			get
			{
				return this._orderUrgencyText;
			}
			set
			{
				this._orderUrgencyText = value;
			}
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/18/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3247"> 
		///		<ExpectedInput>Common.OrderStatus</ExpectedInput>
		///		<ExpectedOutput>Common.OrderStatus</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="2739"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// OrderStatusCode
		/// </summary>
		public Common.OrderStatus OrderStatusCode
		{
			get
			{
				return this._orderStatusCode;
			}
			set
			{
				this._orderStatusCode = value;
			}
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/18/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3249"> 
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="2740"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// OrderStatusText
		/// </summary>
		public string OrderStatusText
		{
			get
			{
				return this._orderStatusText;
			}
			set
			{
				this._orderStatusText = value;
			}
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/18/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3251"> 
		///		<ExpectedInput>Specimen</ExpectedInput>
		///		<ExpectedOutput>Specimen</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="2741"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Specimen
		/// </summary>
		public Specimen Specimen
		{
			get
			{
				return this._specimen;
			}
			set
			{
				this._specimen = value;
			}
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/18/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3253"> 
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="2742"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// SpecimenStatusText 
		/// </summary>
		public string SpecimenStatusText
		{
			get
			{
				return this._specimenStatusText;
			}
			set
			{
				this._specimenStatusText = value;
			}
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/18/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3255"> 
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="2743"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// VBECSOrderNumber
		/// </summary>
		public string VBECSOrderNumber
		{
			get
			{
				return this._VBECSOrderNumber;
			}
			set
			{
				this._VBECSOrderNumber = value;
			}
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/18/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2744"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="2747"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// LabOrderNumber
		/// </summary>
		public string LabOrderNumber
		{
			get
			{
				return _labOrderNumber;
			}
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/18/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3257"> 
		///		<ExpectedInput>byte[]</ExpectedInput>
		///		<ExpectedOutput>byte[]</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="2748"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// RowVersion
		/// </summary>
		public new byte[] RowVersion
		{
			get
			{
				return this._rowVersion;
			}
			set
			{
				this._rowVersion = value;
			}
		}

		#endregion


		#region methods

		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/9/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2605"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2606"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns a boolean that says if an Ordered Test is Pending
		/// </summary>
		/// <returns></returns>
		public static bool IsOrderedTestPending()
		{
			return DAL.OrderedTest.IsOrderedTestPending();
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/25/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2690"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="2749"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Get the list of all types of tests
		/// </summary>
		/// <returns></returns>
		public static DataTable GetOrderableTestList()
		{
			return DAL.OrderedTest.GetOrderableTestList();
		}

		///<Developers>
		///	<Developer>Margaret Jablonski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/10/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3259"> 
		///		<ExpectedInput>Valid Orderable Test ID</ExpectedInput>
		///		<ExpectedOutput>A data table containing the list of orderable tests</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3260"> 
		///		<ExpectedInput>Invalid Orderable Test ID</ExpectedInput>
		///		<ExpectedOutput>An empty data table</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Get the list of all reorderable tests based on the Parent Order Test ID
		/// </summary>
		/// <param name="orderableTestId"></param>
		/// <returns></returns>
		public static DataTable GetReorderableTestList(int orderableTestId)
		{
			return DAL.OrderedTest.GetReorderableTestList(orderableTestId);
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/25/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2692"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="2759"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Get the pending task list
		/// </summary>
		/// <returns></returns>
		public static DataTable GetTestingTaskList(string orderGroupNumber)
		{
			return DAL.OrderedTest.GetTestingTaskList(orderGroupNumber);
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/25/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2694"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="2760"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Get the list of all task statuses
		/// </summary>
		/// <returns></returns>
		public static DataTable GetTaskStatusList()
		{
			return DAL.OrderedTest.GetTaskStatusList();
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/10/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3004"> 
		///		<ExpectedInput>PatientSpecimentGUID</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3005"> 
		///		<ExpectedInput>Invalid Guid</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Get the list of tasks for specimen
		/// </summary>
		/// <param name="patientSpecimenGuid"></param>
		/// <returns></returns>
		public static DataTable GetOrderedTestsforSpecimen(Guid patientSpecimenGuid)
		{
			return DAL.OrderedTest.GetOrderedTestsforSpecimen(patientSpecimenGuid);
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/10/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3006"> 
		///		<ExpectedInput>OrderedTestGuid</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3007"> 
		///		<ExpectedInput>Invalid OrderedTestGuid</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns a data table of QC results
		/// </summary>
		/// <param name="orderedTestGuid"></param>
		/// <returns></returns>
		public static DataTable GetQCResultsForOrderedTest(Guid orderedTestGuid)
		{
			return DAL.OrderedTest.GetQCResultsForOrderedTest(orderedTestGuid);
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/16/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3034"> 
		///		<ExpectedInput>DataSet containing tables with various test results, DateTime of testing, ID of tech, who performed testing, testing method, rack identifier, DivisionCode</ExpectedInput>
		///		<ExpectedOutput>True</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3035"> 
		///		<ExpectedInput>Invalid DataSet, invalid Datetime, invalid tech id, unknown testing method, invalid rack identifier, invalid DivisionCode</ExpectedInput>
		///		<ExpectedOutput>False</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Converts the DataSet containing patient test results to DataTables that can be
		/// saved into the database
		/// BR_56.05
		/// </summary>
		/// <param name="dsTestResults"></param>
		/// <param name="rackNotTestedExceptions"></param>
		/// <param name="testingDateTime"></param>
		/// <param name="testingTechID"></param>
		/// <param name="testMethod"></param>
		/// <param name="reactivityPhase"></param>
		/// <param name="RackGuid"></param>
		/// <param name="automatedInstrumentName"></param>
		/// <param name="automatedInstrumentQCd"></param>
		/// <param name="lastUpdateFunctionId">UC or calling Method</param>
		/// <param name="activePatientGuid">Patient Guid coming from the antibody id control</param>
		/// <param name="workloadProcessIds">All of the workload process ids that the user selected in UC_57</param>
		/// <param name="expiredOrderExceptions"></param>
		/// <param name="bNoWorkloadDefinedMsgNeeded"></param>
		/// <returns></returns>
		public static bool SaveTestResults(DataSet dsTestResults, ArrayList rackNotTestedExceptions,  
			DateTime testingDateTime, string testingTechID, Common.TestingMethod testMethod, Common.ReactivityPhase reactivityPhase, Guid RackGuid, string automatedInstrumentName,
			bool automatedInstrumentQCd, Common.UpdateFunction lastUpdateFunctionId, ArrayList workloadProcessIds, Guid activePatientGuid, ArrayList expiredOrderExceptions, ref bool bNoWorkloadDefinedMsgNeeded)
		{
			DataRow drUpdateOrderedTests;
			DataSet dsSave = new DataSet();
			DataTable dtSpecimenTestsInsert = BOL.SpecimenTest.InitializeSpecimenTestTable(false);
			DataTable dtSpecimenTestsUpdate = BOL.SpecimenTest.InitializeSpecimenTestTable(true);
			DataTable dtTransfusionRequirements = BOL.PatientTransfusionRequirement.InitializeTransfusionRequirementsTable(false);

			
			DataTable dtOrderedTestForVista = null;
			DataTable dtBloodUnitTestsInsert = BOL.BloodUnitTest.GetBloodUnitTestTableForXM(false);
			DataTable dtBloodUnitTestsUpdate = BOL.BloodUnitTest.GetBloodUnitTestTableForXM(true);

			DataTable dtRackLots = InitializeRackLotTable();
			DataTable dtRackTestsInsert = InitializeRackTestTable(false);
			DataTable dtRackTestsUpdate = InitializeRackTestTable(true);
			DataTable dtAntiseraOrderedTests = InitializeAntiseraOrderedTestTable();
			DataTable dtUpdateOrderedTests = InitializeOrderedTestForUpdateTable();
			DataTable dtUpdateOrderedUnits = InitializeOrderedUnitForUpdateTable();

			DataTable dtExpiredAntiseraExceptions = null;
			DataTable dtRackNotTestedExceptions = null;
			DataTable dtAboRhDiscrepancyExceptions = null;
			DataTable dtDiscrepantAntigenTypingExceptions = null;
			DataTable dtAboRhInconclusiveExceptions = null;
			DataTable dtPatientTestExceptions = null;
			DataTable dtUnitSelectionExceptions = null;
			DataTable dtEmergencyIssueUnitsTestingProblemExceptions = null;
			DataTable dtExpiredOrderExceptions = null;
			DataTable dtInconclusiveCrossmatchExceptions = null;
			DataTable dtWorkloadEvents = null;
			Guid [] patientGuids = null;
			Common.WorkloadProcessID workloadProcessId = Common.WorkloadProcessID.Unknown;
			bool partiallyCompleted = false;
			System.Data.DataTable dt = null;
		
			patientGuids = new Guid[1];
			patientGuids[0] = activePatientGuid;
			//looping through all test results and convering them to DataTables that are saved to database
			foreach(DataTable dtTestResults in dsTestResults.Tables)
			{
				if (dtTestResults.TableName == ARTIFICIAL.ABORh)
				{					
					
					if (Convert.ToInt32(dtTestResults.Rows[0][TABLE.OrderedTest.OrderableTestId]) == (int)Common.OrderableTest.ABORh ||
						Convert.ToInt32(dtTestResults.Rows[0][TABLE.OrderedTest.OrderableTestId]) == (int)Common.OrderableTest.TAS)
					{
						workloadProcessId = Common.WorkloadProcessID.ABOForwardReverseTypingPatient;
					}
					else
					{
						workloadProcessId = Common.WorkloadProcessID.ABOForwardReverseTypingPatientRepeatTest;
					}

					Guid orderedTestGuid = (Guid)dtTestResults.Rows[0][TABLE.OrderedTest.OrderedTestGuid];
					partiallyCompleted = (bool)dtTestResults.Rows[0][ARTIFICIAL.PartiallyCompleted];

					if(!partiallyCompleted &&
						((bool)dtTestResults.Rows[0][ARTIFICIAL.AboInterpSave] == true ||
						(bool)dtTestResults.Rows[0][ARTIFICIAL.RhInterpSave] == true))
					{
						dt  = DAL.WorkloadEvent.GetWorkloadProcessGuidByProcessID((int) workloadProcessId);
					
						if(dt!= null)
						{
							if(dt.Rows.Count == 0)
							{
								bNoWorkloadDefinedMsgNeeded = true;
							}
						}

						//Create the workload datatable
						if(dtWorkloadEvents == null)
						{					
							dtWorkloadEvents = BOL.WorkloadEvent.GenerateWorkloadData(workloadProcessId, orderedTestGuid, Guid.Empty, patientGuids, Common.WorkloadTransactionType.Patient, false);
						}
						else
						{
						
							DataTable dtWorkloadEventsTemp = BOL.WorkloadEvent.GenerateWorkloadData(workloadProcessId, orderedTestGuid, Guid.Empty, patientGuids, Common.WorkloadTransactionType.Patient, false);
							foreach(DataRow dtRow in dtWorkloadEventsTemp.Rows)
							{
                                //*** Fortify Justified Code ***
                                //dtWorkloadEvents can't be null since it is checked earler (see above). 
                                DataRow newRow = dtWorkloadEvents.NewRow();
                                //*** Fortify Justified Code ***
                                newRow.ItemArray = dtRow.ItemArray;
								dtWorkloadEvents.Rows.Add(newRow);
							}
						}
					}
					
					//separating exceptions
					foreach(DataRow drTestResult in dtTestResults.Rows)
					{
						foreach (DataColumn dtCol in dtTestResults.Columns)
						{
							if (dtCol.ColumnName.IndexOf("EmergencyIssueExceptions") != -1)
							{
								if (drTestResult.IsNull(dtCol) == false)
								{
									ArrayList emergencyIssueExceptions = (ArrayList)drTestResult[dtCol];
									if (dtEmergencyIssueUnitsTestingProblemExceptions == null)
									{
										dtEmergencyIssueUnitsTestingProblemExceptions = ((BOL.ExceptionReport)emergencyIssueExceptions[0]).ExceptionData.Table.Clone();
										dtEmergencyIssueUnitsTestingProblemExceptions.TableName = TABLE.ExceptionEmergencyIssueUnitTest.TableName;
									}
                                    //*** Fortify Justified Code ***
                                    //dtEmergencyIssueUnitsTestingProblemExceptions can't be null since it is checked earler (see above). 
                                    foreach (ExceptionReport exRep in emergencyIssueExceptions)
										dtEmergencyIssueUnitsTestingProblemExceptions.Rows.Add(exRep.ExceptionData.ItemArray);
                                    //*** Fortify Justified Code ***
                                }
                            }
						}
					}

					BOL.SpecimenTest.BuildSpecimenTestRowsFromABORh(dtTestResults, dtSpecimenTestsInsert, dtSpecimenTestsUpdate, testingDateTime, testingTechID, automatedInstrumentName, automatedInstrumentQCd);
					if (dtTestResults.Rows[0].IsNull(ARTIFICIAL.AboRhInconclusiveException) == false)
					{
						if (dtAboRhInconclusiveExceptions == null)
						{
							dtAboRhInconclusiveExceptions = ((BOL.ExceptionReport)dtTestResults.Rows[0][ARTIFICIAL.AboRhInconclusiveException]).ExceptionData.Table.Clone();
						}
                        //*** Fortify Justified Code ***
                        //dtAboRhInconclusiveExceptions can't be null since it is checked earler (see above). 
                        dtAboRhInconclusiveExceptions.Rows.Add(((BOL.ExceptionReport)dtTestResults.Rows[0][ARTIFICIAL.AboRhInconclusiveException]).ExceptionData.ItemArray);
                        //*** Fortify Justified Code ***
                    }

                    if (dtTestResults.Rows[0].IsNull(ARTIFICIAL.AboRhDiscrepancyException) == false)
					{
						if (dtAboRhDiscrepancyExceptions == null)
						{
							dtAboRhDiscrepancyExceptions = ((BOL.ExceptionReport)dtTestResults.Rows[0][ARTIFICIAL.AboRhDiscrepancyException]).ExceptionData.Table.Clone();
						}
                        //*** Fortify Justified Code ***
                        //dtAboRhDiscrepancyExceptions can't be null since it is checked earler (see above). 
                        dtAboRhDiscrepancyExceptions.Rows.Add(((BOL.ExceptionReport)dtTestResults.Rows[0][ARTIFICIAL.AboRhDiscrepancyException]).ExceptionData.ItemArray);
                        //*** Fortify Justified Code ***
                    }

                    if (dtTestResults.Rows[0].IsNull(ARTIFICIAL.RepeatAboRhDiscrepancyException) == false)
					{
						if (dtAboRhDiscrepancyExceptions == null)
						{
							dtAboRhDiscrepancyExceptions = ((BOL.ExceptionReport)dtTestResults.Rows[0][ARTIFICIAL.RepeatAboRhDiscrepancyException]).ExceptionData.Table.Clone();
						}
                        //*** Fortify Justified Code ***
                        //dtAboRhDiscrepancyExceptions can't be null since it is checked earler (see above). 
                        dtAboRhDiscrepancyExceptions.Rows.Add(((BOL.ExceptionReport)dtTestResults.Rows[0][ARTIFICIAL.RepeatAboRhDiscrepancyException]).ExceptionData.ItemArray);
                        //*** Fortify Justified Code ***
                    }
                }
				else if (dtTestResults.TableName == "ABS")
				{
					if (Convert.ToInt32(dtTestResults.Rows[0][TABLE.OrderedTest.OrderableTestId]) == (int)Common.OrderableTest.ABS ||
						Convert.ToInt32(dtTestResults.Rows[0][TABLE.OrderedTest.OrderableTestId]) == (int)Common.OrderableTest.TAS)
					{
						workloadProcessId = Common.WorkloadProcessID.AntibodyScreenPatient;
					}
					else
					{
						workloadProcessId = Common.WorkloadProcessID.AntibodyScreenPatientRepeatTest;
					}

					partiallyCompleted = (bool)dtTestResults.Rows[0][ARTIFICIAL.PartiallyCompleted];
					Guid orderedTestGuid = (Guid)dtTestResults.Rows[0][TABLE.OrderedTest.OrderedTestGuid];
					
					if(!partiallyCompleted &&
						(bool)dtTestResults.Rows[0][ARTIFICIAL.ABSInterpSave] == true)
					{
						dt  = DAL.WorkloadEvent.GetWorkloadProcessGuidByProcessID((int) workloadProcessId);
					
						if(dt!= null)
						{
							if(dt.Rows.Count == 0)
							{
								bNoWorkloadDefinedMsgNeeded = true;
							}
						}

						if(dtWorkloadEvents == null)
						{					
							dtWorkloadEvents = BOL.WorkloadEvent.GenerateWorkloadData(workloadProcessId, orderedTestGuid, Guid.Empty, patientGuids, Common.WorkloadTransactionType.Patient, false);
						}
						else
						{
						
							DataTable dtWorkloadEventsTemp = BOL.WorkloadEvent.GenerateWorkloadData(workloadProcessId, orderedTestGuid, Guid.Empty, patientGuids, Common.WorkloadTransactionType.Patient, false);
							foreach(DataRow dtRow in dtWorkloadEventsTemp.Rows)
							{
                                //*** Fortify Justified Code ***
                                //dtWorkloadEvents can't be null since it is checked earler (see above). 
                                DataRow newRow = dtWorkloadEvents.NewRow();
                                //*** Fortify Justified Code ***
                                newRow.ItemArray = dtRow.ItemArray;
								dtWorkloadEvents.Rows.Add(newRow);
							}
						}
					}

					BOL.SpecimenTest.BuildSpecimenTestRowsFromABS(dtTestResults, dtSpecimenTestsInsert, dtSpecimenTestsUpdate, testingDateTime, testingTechID, reactivityPhase, automatedInstrumentName, automatedInstrumentQCd);
					
					//separating exceptions
					foreach(DataRow drTestResult in dtTestResults.Rows)
					{
						foreach (DataColumn dtCol in dtTestResults.Columns)
						{
							if (drTestResult[dtCol].GetType().ToString().IndexOf("ExceptionReport") != -1)
							{
								if (drTestResult.IsNull(dtCol) == false)
								{
									if (dtPatientTestExceptions == null)
									{
										dtPatientTestExceptions = ((BOL.ExceptionReport)drTestResult[dtCol]).ExceptionData.Table.Clone();
										dtPatientTestExceptions.TableName = TABLE.ExceptionPatientTest.TableName;
									}
                                    //*** Fortify Justified Code ***
                                    //dtPatientTestExceptions can't be null since it is checked earler (see above). 
                                    dtPatientTestExceptions.Rows.Add(((BOL.ExceptionReport)drTestResult[dtCol]).ExceptionData.ItemArray);
                                    //*** Fortify Justified Code ***
                                }
                            }

							if (dtCol.ColumnName.IndexOf("EmergencyIssueExceptions") != -1)
							{
								if (drTestResult.IsNull(dtCol) == false)
								{
									ArrayList emergencyIssueExceptions = (ArrayList)drTestResult[dtCol];
									if (dtEmergencyIssueUnitsTestingProblemExceptions == null)
									{
										dtEmergencyIssueUnitsTestingProblemExceptions = ((BOL.ExceptionReport)emergencyIssueExceptions[0]).ExceptionData.Table.Clone();
										dtEmergencyIssueUnitsTestingProblemExceptions.TableName = TABLE.ExceptionEmergencyIssueUnitTest.TableName;
									}

                                    //*** Fortify Justified Code ***
                                    //dtEmergencyIssueUnitsTestingProblemExceptions can't be null since it is checked earler (see above). 
                                    foreach (ExceptionReport exRep in emergencyIssueExceptions)
										dtEmergencyIssueUnitsTestingProblemExceptions.Rows.Add(exRep.ExceptionData.ItemArray);
                                    //*** Fortify Justified Code ***
                                }
                            }
						}
					}
				}
				else if (dtTestResults.TableName == "DAT" ||
					dtTestResults.TableName == "DATIgG" ||
					dtTestResults.TableName == "DATC3d")
				{
					if (Convert.ToInt32(dtTestResults.Rows[0][TABLE.OrderedTest.OrderableTestId]) == (int)Common.OrderableTest.DAT ||
						Convert.ToInt32(dtTestResults.Rows[0][TABLE.OrderedTest.OrderableTestId]) == (int)Common.OrderableTest.TAS ||
						Convert.ToInt32(dtTestResults.Rows[0][TABLE.OrderedTest.OrderableTestId]) == (int)Common.OrderableTest.DATIgG ||
						Convert.ToInt32(dtTestResults.Rows[0][TABLE.OrderedTest.OrderableTestId]) == (int)Common.OrderableTest.DATComp ||
						Convert.ToInt32(dtTestResults.Rows[0][TABLE.OrderedTest.OrderableTestId]) == (int)Common.OrderableTest.ABS)
					{
						workloadProcessId = Common.WorkloadProcessID.DirectAntiglobulinTestDAT;
					}
					else
					{
						workloadProcessId = Common.WorkloadProcessID.DirectAntiglobulinTestDATRepeatTest;
					}

					partiallyCompleted = (bool)dtTestResults.Rows[0][ARTIFICIAL.PartiallyCompleted];
					bool DATInterpSave = false;
					if (dtTestResults.TableName == "DAT")
					{
						DATInterpSave = (bool)dtTestResults.Rows[0][ARTIFICIAL.AHGInterpSave];
					}
					else if (dtTestResults.TableName == "DATIgG")
					{
						DATInterpSave = (bool)dtTestResults.Rows[0][ARTIFICIAL.IgGInterpSave];
					}
					else if (dtTestResults.TableName == "DATC3d")
					{
						DATInterpSave = (bool)dtTestResults.Rows[0][ARTIFICIAL.C3dInterpSave];
					}

					Guid orderedTestGuid = (Guid)dtTestResults.Rows[0][TABLE.OrderedTest.OrderedTestGuid];
					if(!partiallyCompleted &&
						DATInterpSave == true)
					{
						dt  = DAL.WorkloadEvent.GetWorkloadProcessGuidByProcessID((int) workloadProcessId);
					
						if(dt!= null)
						{
							if(dt.Rows.Count == 0)
							{
								bNoWorkloadDefinedMsgNeeded = true;
							}
						}

						//Create the workload datatable
						if(dtWorkloadEvents == null)
						{					
							dtWorkloadEvents = BOL.WorkloadEvent.GenerateWorkloadData(workloadProcessId,  orderedTestGuid, Guid.Empty, patientGuids, Common.WorkloadTransactionType.Patient, false);
						}
						else
						{
					
							DataTable dtWorkloadEventsTemp = BOL.WorkloadEvent.GenerateWorkloadData(workloadProcessId,  orderedTestGuid, Guid.Empty, patientGuids, Common.WorkloadTransactionType.Patient, false);
							foreach(DataRow dtRow in dtWorkloadEventsTemp.Rows)
							{
                                //*** Fortify Justified Code ***
                                //dtWorkloadEvents can't be null since it is checked earler (see above). 
                                DataRow newRow = dtWorkloadEvents.NewRow();
                                //*** Fortify Justified Code ***
                                newRow.ItemArray = dtRow.ItemArray;
								dtWorkloadEvents.Rows.Add(newRow);
							}
						}

						if (dtTestResults.Rows.Count > 1)
						{
							//checking second workload for DAT
							dt  = DAL.WorkloadEvent.GetWorkloadProcessGuidByProcessID((int) Common.WorkloadProcessID.DATQC);
					
							if(dt!= null)
							{
								if(dt.Rows.Count == 0)
								{
									bNoWorkloadDefinedMsgNeeded = true;
								}
							}
							//creating second workload for DAT (DATQC)
							if(dtWorkloadEvents == null)
							{					
								dtWorkloadEvents = BOL.WorkloadEvent.GenerateWorkloadData(Common.WorkloadProcessID.DATQC,  orderedTestGuid, Guid.Empty, patientGuids, Common.WorkloadTransactionType.Miscellanious, false);
							}
							else
							{
					
								DataTable dtWorkloadEventsTemp = BOL.WorkloadEvent.GenerateWorkloadData(Common.WorkloadProcessID.DATQC,  orderedTestGuid, Guid.Empty, patientGuids, Common.WorkloadTransactionType.Miscellanious, false);
								foreach(DataRow dtRow in dtWorkloadEventsTemp.Rows)
								{
                                    //*** Fortify Justified Code ***
                                    //dtWorkloadEvents can't be null since it is checked earler (see above). 
                                    DataRow newRow = dtWorkloadEvents.NewRow();
                                    //*** Fortify Justified Code ***
                                    newRow.ItemArray = dtRow.ItemArray;
									dtWorkloadEvents.Rows.Add(newRow);
								}
							}
						}
					}

					//separating exceptions
					foreach(DataRow drTestResult in dtTestResults.Rows)
					{
						foreach (DataColumn dtCol in dtTestResults.Columns)
						{
							if (drTestResult[dtCol].GetType().ToString().IndexOf("ExceptionReport") != -1)
							{
								if (drTestResult.IsNull(dtCol) == false)
								{
									if (dtExpiredAntiseraExceptions == null)
									{
										dtExpiredAntiseraExceptions = ((BOL.ExceptionReport)drTestResult[dtCol]).ExceptionData.Table.Clone();
										dtExpiredAntiseraExceptions.TableName = TABLE.ExceptionQc.TableName;
									}
                                    //*** Fortify Justified Code ***
                                    //dtExpiredAntiseraExceptions can't be null since it is checked earler (see above). 
                                    dtExpiredAntiseraExceptions.Rows.Add(((BOL.ExceptionReport)drTestResult[dtCol]).ExceptionData.ItemArray);
                                    //*** Fortify Justified Code ***
                                }
                            }

							if (dtCol.ColumnName.IndexOf("EmergencyIssueExceptions") != -1)
							{
								if (drTestResult.IsNull(dtCol) == false)
								{
									ArrayList emergencyIssueExceptions = (ArrayList)drTestResult[dtCol];
									if (dtEmergencyIssueUnitsTestingProblemExceptions == null)
									{
										dtEmergencyIssueUnitsTestingProblemExceptions = ((BOL.ExceptionReport)emergencyIssueExceptions[0]).ExceptionData.Table.Clone();
										dtEmergencyIssueUnitsTestingProblemExceptions.TableName = TABLE.ExceptionEmergencyIssueUnitTest.TableName;
									}

                                    //*** Fortify Justified Code ***
                                    //dtEmergencyIssueUnitsTestingProblemExceptions can't be null since it is checked earler (see above). 
                                    foreach (ExceptionReport exRep in emergencyIssueExceptions)
										dtEmergencyIssueUnitsTestingProblemExceptions.Rows.Add(exRep.ExceptionData.ItemArray);
                                    //*** Fortify Justified Code ***
                                }
                            }
						}
					}

					BOL.SpecimenTest.BuildSpecimenTestRowsFromDAT(dtTestResults, dtSpecimenTestsInsert, dtSpecimenTestsUpdate, testingDateTime, testingTechID, automatedInstrumentName, automatedInstrumentQCd);

					if (new BOL.Division(Common.LogonUser.LogonUserDivisionCode).ServiceTypeIndicator == true ||
						partiallyCompleted == true)
					{
						BOL.SpecimenTest.BuildQCDataFromDAT(dtTestResults, dtRackLots, dtRackTestsInsert, dtRackTestsUpdate, dtAntiseraOrderedTests, testingDateTime, testingTechID);
					}
				}
				else if (dtTestResults.TableName == "ABID")
				{
					Guid orderedTestGuid = (Guid)dtTestResults.Rows[0][TABLE.OrderedTest.OrderedTestGuid];
					BOL.SpecimenTest.BuildSpecimenTestRowsFromABID(dtTestResults, dtSpecimenTestsInsert, dtTransfusionRequirements, testingDateTime, testingTechID, RackGuid, automatedInstrumentName, automatedInstrumentQCd);
					if(workloadProcessIds != null)
					{
						if (workloadProcessIds.Count > 0)
						{
							for(int nCtr=0; nCtr<workloadProcessIds.Count; nCtr=nCtr+2)
							{
								dt  = DAL.WorkloadEvent.GetWorkloadProcessGuidByProcessID((int) (Common.WorkloadProcessID) workloadProcessIds[nCtr]);
								if(dt!= null)
								{
									if(dt.Rows.Count == 0)
									{
										bNoWorkloadDefinedMsgNeeded = true;
									}
								}

								if(dtWorkloadEvents == null)
								{					
									dtWorkloadEvents = BOL.WorkloadEvent.GenerateWorkloadData((Common.WorkloadProcessID) workloadProcessIds[nCtr], (int) workloadProcessIds[nCtr+1], orderedTestGuid, Guid.Empty, patientGuids, Common.WorkloadTransactionType.Patient, false);
								}
								else
								{
						
									DataTable dtWorkloadEventsTemp = BOL.WorkloadEvent.GenerateWorkloadData((Common.WorkloadProcessID) workloadProcessIds[nCtr], (int) workloadProcessIds[nCtr+1], orderedTestGuid, Guid.Empty, patientGuids, Common.WorkloadTransactionType.Patient, false);
									foreach(DataRow dtRow in dtWorkloadEventsTemp.Rows)
									{
                                        //*** Fortify Justified Code ***
                                        //dtWorkloadEvents can't be null since it is checked earler (see above). 
                                        DataRow newRow = dtWorkloadEvents.NewRow();
                                        //*** Fortify Justified Code ***
                                        newRow.ItemArray = dtRow.ItemArray;
										dtWorkloadEvents.Rows.Add(newRow);
									}
								}
							}
						}
					}
				}
				else if (dtTestResults.TableName.Substring(0,2) == "AG")
				{
					Common.WorkloadProcessID secondWorkloadProcessId;

					if (Convert.ToInt32(dtTestResults.Rows[0][TABLE.ReactivityPhase.ReactivityPhaseCode]) == (int)Common.ReactivityPhase.AHGandCC ||
						Convert.ToInt32(dtTestResults.Rows[0][TABLE.ReactivityPhase.ReactivityPhaseCode]) == (int)Common.ReactivityPhase.IS)
					{
						//capture workload for Patient antigen phenotype (single phase)
						workloadProcessId = Common.WorkloadProcessID.PatientAntigenPhenotypeSinglePhase;
						secondWorkloadProcessId = Common.WorkloadProcessID.AntigenPhenotypingSingleTestPhaseQC;
					}
					else
					{
						//capture workload for Patient antigen phenotype (multiple phases)
						workloadProcessId = Common.WorkloadProcessID.PatientAntigenPhenotypeMultiplePhases;
						secondWorkloadProcessId = Common.WorkloadProcessID.AntigenPhenotypingMultipleTestPhasesQC;
					}

					partiallyCompleted = (bool)dtTestResults.Rows[0][ARTIFICIAL.PartiallyCompleted];
					Guid orderedTestGuid = (Guid)dtTestResults.Rows[0][TABLE.OrderedTest.OrderedTestGuid];
					if(!partiallyCompleted)
					{
						dt  = DAL.WorkloadEvent.GetWorkloadProcessGuidByProcessID((int) workloadProcessId);
					
						if(dt!= null)
						{
							if(dt.Rows.Count == 0)
							{
								bNoWorkloadDefinedMsgNeeded = true;
							}
						}

						//Create the workload datatable
						if(dtWorkloadEvents == null)
						{					
							dtWorkloadEvents = BOL.WorkloadEvent.GenerateWorkloadData(workloadProcessId, orderedTestGuid, Guid.Empty, patientGuids, Common.WorkloadTransactionType.Patient, false);
						}
						else
						{
						
							DataTable dtWorkloadEventsTemp = BOL.WorkloadEvent.GenerateWorkloadData(workloadProcessId, orderedTestGuid, Guid.Empty, patientGuids, Common.WorkloadTransactionType.Patient, false);
							foreach(DataRow dtRow in dtWorkloadEventsTemp.Rows)
							{
                                //*** Fortify Justified Code ***
                                //dtWorkloadEvents can't be null since it is checked earler (see above). 
                                DataRow newRow = dtWorkloadEvents.NewRow();
                                //*** Fortify Justified Code ***
                                newRow.ItemArray = dtRow.ItemArray;
								dtWorkloadEvents.Rows.Add(newRow);
							}
						}

						if (dtTestResults.Rows.Count == 3)
						{
							//second workload for AG (QC)
							dt  = DAL.WorkloadEvent.GetWorkloadProcessGuidByProcessID((int) secondWorkloadProcessId);
					
							if(dt!= null)
							{
								if(dt.Rows.Count == 0)
								{
									bNoWorkloadDefinedMsgNeeded = true;
								}
							}
						
							//Create the secong workload datatable (QC)
							if(dtWorkloadEvents == null)
							{					
								dtWorkloadEvents = BOL.WorkloadEvent.GenerateWorkloadData(secondWorkloadProcessId, orderedTestGuid, Guid.Empty, patientGuids, Common.WorkloadTransactionType.Miscellanious, false);
							}
							else
							{
						
								DataTable dtWorkloadEventsTemp = BOL.WorkloadEvent.GenerateWorkloadData(secondWorkloadProcessId, orderedTestGuid, Guid.Empty, patientGuids, Common.WorkloadTransactionType.Miscellanious, false);
								foreach(DataRow dtRow in dtWorkloadEventsTemp.Rows)
								{
                                    //*** Fortify Justified Code ***
                                    //dtWorkloadEvents can't be null since it is checked earler (see above). 
                                    DataRow newRow = dtWorkloadEvents.NewRow();
                                    //*** Fortify Justified Code ***
                                    newRow.ItemArray = dtRow.ItemArray;
									dtWorkloadEvents.Rows.Add(newRow);
								}
							}
						}

					}
					//separating exceptions
					foreach(DataRow drTestResult in dtTestResults.Rows)
					{
						foreach (DataColumn dtCol in dtTestResults.Columns)
						{
							if (drTestResult[dtCol].GetType().ToString().IndexOf("ExceptionReport") != -1)
							{
								if (drTestResult.IsNull(dtCol) == false)
								{
									if (dtCol.ColumnName == "ReagentExpiredException")
									{
										if (dtExpiredAntiseraExceptions == null)
										{
											dtExpiredAntiseraExceptions = ((BOL.ExceptionReport)drTestResult[dtCol]).ExceptionData.Table.Clone();
											dtExpiredAntiseraExceptions.TableName = TABLE.ExceptionQc.TableName;
										}
                                        //*** Fortify Justified Code ***
                                        //dtExpiredAntiseraExceptions can't be null since it is checked earler (see above). 
                                        dtExpiredAntiseraExceptions.Rows.Add(((BOL.ExceptionReport)drTestResult[dtCol]).ExceptionData.ItemArray);
                                        //*** Fortify Justified Code ***
                                    }
                                    else if (dtCol.ColumnName == "DiscrepantTypingException")
									{
										if (dtDiscrepantAntigenTypingExceptions == null)
										{
											dtDiscrepantAntigenTypingExceptions = ((BOL.ExceptionReport)drTestResult[dtCol]).ExceptionData.Table.Clone();
											dtDiscrepantAntigenTypingExceptions.TableName = TABLE.ExceptionPatientAntigenDiscrepancy.TableName;
										}
                                        //*** Fortify Justified Code ***
                                        //dtDiscrepantAntigenTypingExceptions can't be null since it is checked earler (see above). 
                                        dtDiscrepantAntigenTypingExceptions.Rows.Add(((BOL.ExceptionReport)drTestResult[dtCol]).ExceptionData.ItemArray);
                                        //*** Fortify Justified Code ***
                                    }
                                    else if (dtCol.ColumnName == "PhaseChangedException")
									{
										if (dtPatientTestExceptions == null)
										{
											dtPatientTestExceptions = ((BOL.ExceptionReport)drTestResult[dtCol]).ExceptionData.Table.Clone();
											dtPatientTestExceptions.TableName = TABLE.ExceptionPatientTest.TableName;
										}
                                        //*** Fortify Justified Code ***
                                        //dtPatientTestExceptions can't be null since it is checked earler (see above). 
                                        dtPatientTestExceptions.Rows.Add(((BOL.ExceptionReport)drTestResult[dtCol]).ExceptionData.ItemArray);
                                        //*** Fortify Justified Code ***
                                    }
                                }
							}
						}
					}

					BOL.SpecimenTest.BuildSpecimenTestRowsFromAGT(dtTestResults, dtSpecimenTestsInsert, dtSpecimenTestsUpdate, testingDateTime, testingTechID, testMethod, automatedInstrumentName, automatedInstrumentQCd);

					if (new BOL.Division(Common.LogonUser.LogonUserDivisionCode).ServiceTypeIndicator == true ||
						partiallyCompleted == true)
					{
						BOL.SpecimenTest.BuildQCDataFromAGT(dtTestResults, dtRackLots, dtRackTestsInsert, dtRackTestsUpdate, dtAntiseraOrderedTests, testingDateTime, testingTechID);
					}
				}
				else if (dtTestResults.TableName == "XM")
				{
					ArrayList orderedTestGuids = new ArrayList();
					ArrayList orderedComponentGuids = new ArrayList();

					foreach (DataRow dr in dtTestResults.Rows)
					{
						if(dtTestResults.Rows[0].IsNull(TABLE.OrderedTest.OrderedTestGuid))
						{
							if (orderedComponentGuids.IndexOf((Guid)dr[TABLE.OrderedComponent.OrderedComponentGuid]) == -1)
								orderedComponentGuids.Add((Guid)dr[TABLE.OrderedComponent.OrderedComponentGuid]);
						}
						else
						{
							if (orderedTestGuids.IndexOf((Guid)dr[TABLE.OrderedTest.OrderedTestGuid]) == -1)
								orderedTestGuids.Add((Guid)dr[TABLE.OrderedTest.OrderedTestGuid]);
						}

						if (dr[ARTIFICIAL.XMInterp].ToString().Trim() != string.Empty)
						{
							if (dr[ARTIFICIAL.IS].ToString().Trim() != string.Empty &&
								dr[ARTIFICIAL.C37].ToString().Trim() == "X" &&
								dr[ARTIFICIAL.AHG].ToString().Trim() == "X" &&
								dr[ARTIFICIAL.CC].ToString().Trim() == "X")
							{
								//capture workload for Patient Crossmatch serologic IS
								//check OrderedTest and OrderedComponent Guids
								workloadProcessId = Common.WorkloadProcessID.CrossmatchUnitSerologicImmediateSpin;
							}
							else
							{
								//capture workload for Patient Crossmatch serologic Coombs
								//check OrderedTest and OrderedComponent Guids
								workloadProcessId = Common.WorkloadProcessID.CrossmatchUnitSerologicalCoombs;
							}

							Guid [] orderedTestGuidsArray = new Guid[orderedTestGuids.Count];
							Guid [] orderedComponentGuidsArray = new Guid[orderedComponentGuids.Count];
							Guid [] patientGuidsArray = new Guid[orderedComponentGuids.Count];
							//CR 2212
							int count = 0;
							foreach(Guid orderedTestGuid in orderedTestGuids)
							{
								orderedTestGuidsArray[count] = orderedTestGuid;
								dtWorkloadEvents = BOL.WorkloadEvent.GenerateWorkloadData(workloadProcessId, orderedTestGuid, System.Guid.Empty, patientGuidsArray, Common.WorkloadTransactionType.Patient, false);
								count++;

							}

							foreach(Guid orderedComponentGuid in orderedComponentGuids)
							{
								patientGuidsArray[count] = activePatientGuid;
								orderedComponentGuidsArray[count] = orderedComponentGuid;
								if(dtWorkloadEvents == null)
								{
									dtWorkloadEvents = BOL.WorkloadEvent.GenerateWorkloadData(workloadProcessId,System.Guid.Empty, orderedComponentGuid,  patientGuidsArray, Common.WorkloadTransactionType.Patient, false);
								}
								else
								{
									DataTable dtWorkloadEventsTemp = BOL.WorkloadEvent.GenerateWorkloadData(workloadProcessId, System.Guid.Empty, orderedComponentGuid,  patientGuidsArray, Common.WorkloadTransactionType.Patient, false);
									foreach(DataRow dtRow in dtWorkloadEventsTemp.Rows)
									{
                                        //*** Fortify Justified Code ***
                                        //dtWorkloadEvents can't be null since it is checked earler (see above). 
                                        DataRow newRow = dtWorkloadEvents.NewRow();
                                        //*** Fortify Justified Code ***
                                        newRow.ItemArray = dtRow.ItemArray;
										dtWorkloadEvents.Rows.Add(newRow);
									}
								}
								count++;
							}

							//adding workloads for crossmatch
							if (orderedComponentGuidsArray.Length > 0)
							{
								//check if workload is defined
								dt  = DAL.WorkloadEvent.GetWorkloadProcessGuidByProcessID((int) workloadProcessId);
					
								if(dt!= null)
								{
									if(dt.Rows.Count == 0)
									{
										bNoWorkloadDefinedMsgNeeded = true;
									}
								}

							}

							//adding workloads for repeat crossmatch
							if (orderedTestGuidsArray.Length > 0)
							{
								//check if workload for repeat XM is defined
								dt  = DAL.WorkloadEvent.GetWorkloadProcessGuidByProcessID((int) Common.WorkloadProcessID.CrossmatchRepeatTest);
					
								if(dt!= null)
								{
									if(dt.Rows.Count == 0)
									{
										bNoWorkloadDefinedMsgNeeded = true;
									}
								}

								if(dtWorkloadEvents == null)
								{					
									dtWorkloadEvents = BOL.WorkloadEvent.GenerateWorkloadData(Common.WorkloadProcessID.CrossmatchRepeatTest, orderedTestGuidsArray, Common.WorkloadTransactionType.Miscellanious, false);
								}
								else
								{
						
									DataTable dtWorkloadEventsTemp = BOL.WorkloadEvent.GenerateWorkloadData(Common.WorkloadProcessID.CrossmatchRepeatTest, orderedTestGuidsArray, Common.WorkloadTransactionType.Miscellanious, false);
									foreach(DataRow dtRow in dtWorkloadEventsTemp.Rows)
									{
                                        //*** Fortify Justified Code ***
                                        //dtWorkloadEvents can't be null since it is checked earler (see above). 
                                        DataRow newRow = dtWorkloadEvents.NewRow();
                                        //*** Fortify Justified Code ***
                                        newRow.ItemArray = dtRow.ItemArray;
										dtWorkloadEvents.Rows.Add(newRow);
									}
								}	
							}
						}
					}

					BloodUnitTest.BuildBloodUnitTestRowsFromXM(dtTestResults, dtBloodUnitTestsInsert, dtBloodUnitTestsUpdate, dtUpdateOrderedUnits, testingDateTime, testingTechID, reactivityPhase, Common.LogonUser.LogonUserDivisionCode, RackGuid, automatedInstrumentName, lastUpdateFunctionId);
					dsSave.Tables.Add(Common.Utility.AppendLastUpdateInformation(dtBloodUnitTestsInsert, lastUpdateFunctionId));
					dsSave.Tables.Add(Common.Utility.AppendLastUpdateInformation(dtBloodUnitTestsUpdate, lastUpdateFunctionId));

					//separating exceptions
					foreach(DataRow drTestResult in dtTestResults.Rows)
					{
						foreach (DataColumn dtCol in dtTestResults.Columns)
						{
							if (drTestResult[dtCol].GetType().ToString().IndexOf("ExceptionReport") != -1)
							{
								if (drTestResult.IsNull(dtCol) == false)
								{
									if (dtCol.ColumnName == "NonStandardTestException")
									{
										if (dtInconclusiveCrossmatchExceptions == null)
										{
											dtInconclusiveCrossmatchExceptions = ((BOL.ExceptionReport)drTestResult[dtCol]).ExceptionData.Table.Clone();
											dtInconclusiveCrossmatchExceptions.TableName = TABLE.ExceptionInconclusiveCrossmatch.TableName;
										}
                                        //*** Fortify Justified Code ***
                                        //dtInconclusiveCrossmatchExceptions can't be null since it is checked earler (see above). 
                                        dtInconclusiveCrossmatchExceptions.Rows.Add(((BOL.ExceptionReport)drTestResult[dtCol]).ExceptionData.ItemArray);
                                        //*** Fortify Justified Code ***
                                    }
                                    else if (dtCol.ColumnName == "EmergencyIssueExceptions")
									{
										if (dtEmergencyIssueUnitsTestingProblemExceptions == null)
										{
											dtEmergencyIssueUnitsTestingProblemExceptions = ((BOL.ExceptionReport)drTestResult[dtCol]).ExceptionData.Table.Clone();
											dtEmergencyIssueUnitsTestingProblemExceptions.TableName = TABLE.ExceptionEmergencyIssueUnitTest.TableName;
										}
										dtEmergencyIssueUnitsTestingProblemExceptions.Rows.Add(((BOL.ExceptionReport)drTestResult[dtCol]).ExceptionData.ItemArray);
									}
									else
									{
										if (dtUnitSelectionExceptions == null)
										{
											dtUnitSelectionExceptions = ((BOL.ExceptionReport)drTestResult[dtCol]).ExceptionData.Table.Clone();
											dtUnitSelectionExceptions.TableName = TABLE.ExceptionUnitSelection.TableName;
										}
                                        //*** Fortify Justified Code ***
                                        //dtUnitSelectionExceptions can't be null since it is checked earler (see above). 
                                        dtUnitSelectionExceptions.Rows.Add(((BOL.ExceptionReport)drTestResult[dtCol]).ExceptionData.ItemArray);
                                        //*** Fortify Justified Code ***
                                    }
                                }
							}
						}
					}
				}

				if (dtTestResults.TableName != TABLE.Patient.TableName &&
					dtTestResults.TableName != "ABIDProcess" &&
					(dtTestResults.TableName != "XM" || dtTestResults.Rows[0].IsNull(TABLE.OrderedTest.OrderedTestGuid) == false))
				{
					if (dtTestResults.Rows[0][TABLE.OrderableTest.OrderableTestId].ToString() == ((int)Common.OrderableTest.TAS).ToString() ||
						dtTestResults.Rows[0][TABLE.OrderableTest.OrderableTestId].ToString() == ((int)Common.OrderableTest.ABS).ToString() ||
						dtTestResults.Rows[0][TABLE.OrderableTest.OrderableTestId].ToString() == ((int)Common.OrderableTest.DAT).ToString())
					{
						if (dtUpdateOrderedTests.Select(TABLE.OrderedTest.OrderedTestGuid + " = '" + dtTestResults.Rows[0][TABLE.OrderedTest.OrderedTestGuid].ToString() + "'").Length == 0)
						{
							drUpdateOrderedTests = dtUpdateOrderedTests.NewRow();
							drUpdateOrderedTests[TABLE.OrderedTest.OrderedTestGuid] = dtTestResults.Rows[0][TABLE.OrderedTest.OrderedTestGuid];
							drUpdateOrderedTests[TABLE.OrderedTest.RowVersion] = dtTestResults.Rows[0][ARTIFICIAL.OrderedTestRowVersion];
							drUpdateOrderedTests[TABLE.OrderedTest.TaskStatusCode] = System.Convert.ToString(Common.Utility.GetTaskStatusCodeFromEnum(Common.TaskStatus.Completed));
						}
						else
						{
							drUpdateOrderedTests = dtUpdateOrderedTests.Select(TABLE.OrderedTest.OrderedTestGuid + " = '" + dtTestResults.Rows[0][TABLE.OrderedTest.OrderedTestGuid].ToString() + "'")[0];
						}
					}
					else
					{
						drUpdateOrderedTests = dtUpdateOrderedTests.NewRow();
						drUpdateOrderedTests[TABLE.OrderedTest.OrderedTestGuid] = dtTestResults.Rows[0][TABLE.OrderedTest.OrderedTestGuid];
						drUpdateOrderedTests[TABLE.OrderedTest.RowVersion] = dtTestResults.Rows[0][ARTIFICIAL.OrderedTestRowVersion];
						drUpdateOrderedTests[TABLE.OrderedTest.TaskStatusCode] = System.Convert.ToString(Common.Utility.GetTaskStatusCodeFromEnum(Common.TaskStatus.Completed));
					}

					//updating TaskStatusCode BR_56.05
					foreach(DataRow drTestResult in dtTestResults.Rows)
					{
						if (dtTestResults.Columns.IndexOf(ARTIFICIAL.PartiallyCompleted) != -1)
						{
							if (drTestResult.IsNull(ARTIFICIAL.PartiallyCompleted) == false)
							{
								if ((bool)drTestResult[ARTIFICIAL.PartiallyCompleted] == true)
								{
									if (dtTestResults.Rows[0][TABLE.OrderableTest.OrderableTestId].ToString() == ((int)Common.OrderableTest.TAS).ToString() ||
										dtTestResults.Rows[0][TABLE.OrderableTest.OrderableTestId].ToString() == ((int)Common.OrderableTest.ABS).ToString() ||
										dtTestResults.Rows[0][TABLE.OrderableTest.OrderableTestId].ToString() == ((int)Common.OrderableTest.DAT).ToString())
									{
										if (drUpdateOrderedTests[TABLE.OrderedTest.TaskStatusCode].ToString() == System.Convert.ToString(Common.Utility.GetTaskStatusCodeFromEnum(Common.TaskStatus.Completed)))
										{
											drUpdateOrderedTests[TABLE.OrderedTest.TaskStatusCode] = System.Convert.ToString(Common.Utility.GetTaskStatusCodeFromEnum(Common.TaskStatus.PartiallyCompleted));
										}
									}
									else
									{
										drUpdateOrderedTests[TABLE.OrderedTest.TaskStatusCode] = System.Convert.ToString(Common.Utility.GetTaskStatusCodeFromEnum(Common.TaskStatus.PartiallyCompleted));
									}
								}
							}
						}
					}

					drUpdateOrderedTests[TABLE.OrderedTest.LastUpdateUser] = Common.LogonUser.LogonUserName;

					//updating OrderStatusCode
					if (drUpdateOrderedTests[TABLE.OrderedTest.TaskStatusCode].ToString() == System.Convert.ToString(Common.Utility.GetTaskStatusCodeFromEnum(Common.TaskStatus.Completed)))
					{
						drUpdateOrderedTests[TABLE.OrderedTest.OrderStatusCode] = System.Convert.ToString(Common.Utility.GetTaskStatusCodeFromEnum(Common.TaskStatus.Completed));
					}
					else
					{
						drUpdateOrderedTests[TABLE.OrderedTest.OrderStatusCode] = DBNull.Value;
					}

					if (dtUpdateOrderedTests.Select(TABLE.OrderedTest.OrderedTestGuid + " = '" + dtTestResults.Rows[0][TABLE.OrderedTest.OrderedTestGuid].ToString() + "'").Length == 0)
					{
						dtUpdateOrderedTests.Rows.Add(drUpdateOrderedTests);
					}
				}
			}

			if (dtExpiredAntiseraExceptions != null)
				dsSave.Tables.Add(dtExpiredAntiseraExceptions);

			if (dtDiscrepantAntigenTypingExceptions != null)
				dsSave.Tables.Add(dtDiscrepantAntigenTypingExceptions);

			if (dtAboRhInconclusiveExceptions != null)
				dsSave.Tables.Add(dtAboRhInconclusiveExceptions);

			if (dtAboRhDiscrepancyExceptions != null)
				dsSave.Tables.Add(dtAboRhDiscrepancyExceptions);

			if (dtPatientTestExceptions != null)
				dsSave.Tables.Add(dtPatientTestExceptions);

			if (dtInconclusiveCrossmatchExceptions != null)
				dsSave.Tables.Add(dtInconclusiveCrossmatchExceptions);

			if (dtUnitSelectionExceptions != null)
				dsSave.Tables.Add(dtUnitSelectionExceptions);

			if (dtEmergencyIssueUnitsTestingProblemExceptions != null)
				dsSave.Tables.Add(dtEmergencyIssueUnitsTestingProblemExceptions);

			if (dtWorkloadEvents != null)
				dsSave.Tables.Add(Common.Utility.AppendLastUpdateInformation(dtWorkloadEvents, lastUpdateFunctionId));

			if (dsSave.Tables.IndexOf(dtRackLots) == -1 &&
				dtRackLots.Rows.Count > 0)
			{
				dsSave.Tables.Add(Common.Utility.AppendLastUpdateInformation(dtRackLots,lastUpdateFunctionId));
			}

			if (dsSave.Tables.IndexOf(dtRackTestsInsert) == -1 &&
				dtRackTestsInsert.Rows.Count > 0)
			{
				dsSave.Tables.Add(Common.Utility.AppendLastUpdateInformation(dtRackTestsInsert,lastUpdateFunctionId));
			}

			if (dsSave.Tables.IndexOf(dtRackTestsUpdate) == -1 &&
				dtRackTestsUpdate.Rows.Count > 0)
			{
				dsSave.Tables.Add(Common.Utility.AppendLastUpdateInformation(dtRackTestsUpdate,lastUpdateFunctionId));
			}

			if (dsSave.Tables.IndexOf(dtAntiseraOrderedTests) == -1 &&
				dtAntiseraOrderedTests.Rows.Count > 0)
			{
				dsSave.Tables.Add(Common.Utility.AppendLastUpdateInformation(dtAntiseraOrderedTests,lastUpdateFunctionId));
			}

			//building RackNotTested Exception DataTable
			if (rackNotTestedExceptions.Count > 0)
			{
				dtRackNotTestedExceptions = ((BOL.ExceptionReport)rackNotTestedExceptions[0]).ExceptionData.Table.Clone();
				dtRackNotTestedExceptions.TableName = TABLE.ExceptionQcNotPerformed.TableName;

				foreach(BOL.ExceptionReport exception in rackNotTestedExceptions)
				{
					DataTable dtPatient = dsTestResults.Tables[TABLE.Patient.TableName];
					if (exception.ExceptionData[TABLE.ExceptionQcNotPerformed.PatientId].ToString() == dtPatient.Rows[0][TABLE.Patient.PatientSsn].ToString() &&
						exception.ExceptionData[TABLE.ExceptionQcNotPerformed.PatientName].ToString() == dtPatient.Rows[0][TABLE.Patient.PatientLastName].ToString() + ", " + dtPatient.Rows[0][TABLE.Patient.PatientFirstName].ToString() &&
						exception.ExceptionData[TABLE.ExceptionQcNotPerformed.SpecimenUid].ToString() == dtPatient.Rows[0][TABLE.PatientSpecimen.SpecimenUid].ToString())
					{
						dtRackNotTestedExceptions.Rows.Add(exception.ExceptionData.ItemArray);
					}
				}
				dsSave.Tables.Add(dtRackNotTestedExceptions);
			}

			//building ExpiredOrder Exception DataTable
			if (expiredOrderExceptions.Count > 0)
			{
				dtExpiredOrderExceptions = ((BOL.ExceptionReport)expiredOrderExceptions[0]).ExceptionData.Table.Clone();
				dtExpiredOrderExceptions.TableName = TABLE.ExceptionExpiredTask.TableName;

				//CR 2805 added check for patient to stop code from crashing
				DataTable dtPatient = dsTestResults.Tables[TABLE.Patient.TableName];
				foreach(BOL.ExceptionReport exception in expiredOrderExceptions)
				{
					//CR 3198
					if (exception.ExceptionData[TABLE.ExceptionExpiredTask.PatientId].ToString() == dtPatient.Rows[0][TABLE.Patient.PatientSsn].ToString().Replace("-","") &&
						exception.ExceptionData[TABLE.ExceptionExpiredTask.SpecimenUid].ToString() == dtPatient.Rows[0][TABLE.PatientSpecimen.SpecimenUid].ToString())
					{
						exception.ExceptionData[TABLE.ExceptionExpiredTask.LoginTechName] = testingTechID;
						dtExpiredOrderExceptions.Rows.Add(exception.ExceptionData.ItemArray);
					}
				}
				dsSave.Tables.Add(dtExpiredOrderExceptions);
			}

			if( !(	dtSpecimenTestsInsert.Rows.Count > 0 ||
				    dtSpecimenTestsUpdate.Rows.Count > 0 ||
					dtBloodUnitTestsInsert.Rows.Count > 0 ||
					dtBloodUnitTestsUpdate.Rows.Count > 0 ||
					dtUpdateOrderedTests.Rows.Count > 0 ||
					dtRackLots.Rows.Count > 0 ||
					dtRackTestsInsert.Rows.Count > 0 ||
					dtRackTestsUpdate.Rows.Count > 0 ||
					dtAntiseraOrderedTests.Rows.Count > 0 ) )
				return true;

			dsSave.Tables.Add(Common.Utility.AppendLastUpdateInformation(dtUpdateOrderedTests,lastUpdateFunctionId));
			dsSave.Tables.Add(Common.Utility.AppendLastUpdateInformation(dtUpdateOrderedUnits,lastUpdateFunctionId));
			dsSave.Tables.Add(Common.Utility.AppendLastUpdateInformation(dtSpecimenTestsInsert,lastUpdateFunctionId));
			dsSave.Tables.Add(Common.Utility.AppendLastUpdateInformation(dtSpecimenTestsUpdate,lastUpdateFunctionId));
			if (dtTransfusionRequirements.Rows.Count > 0)
				dsSave.Tables.Add(Common.Utility.AppendLastUpdateInformation(dtTransfusionRequirements, lastUpdateFunctionId));

			DataTable dtBCEMessages = null;
			DataTable dtTempBCEMessages = null;

			//CR 2941
			//CR 2945
			//generate table with BCE_MESSAGE
			//The Unit is removed from assignment if the crossmatch result is Incompatible

			for (int i=0; i< dtUpdateOrderedUnits.Rows.Count; i++)
			{
				if (dtUpdateOrderedUnits.Rows[i].IsNull(TABLE.OrderedUnit.ReleaseDate) == false)
				{
					Guid bloodUnitGuid = Guid.Empty;
					Guid patientGuid = Guid.Empty;
					Guid orderedUnitGuid = new Guid(dtUpdateOrderedUnits.Rows[i][TABLE.OrderedUnit.OrderedUnitGuid].ToString());
					DataTable orderedUnitDataTable = DAL.OrderedUnit.GetOrderedUnitByGuid(orderedUnitGuid, Common.LogonUser.LogonUserDivisionCode);
					if (orderedUnitDataTable.Rows.Count > 0) 
					{
						bloodUnitGuid = (Guid)orderedUnitDataTable.Rows[0][TABLE.OrderedUnit.BloodUnitGuid];
					}
					patientGuid = activePatientGuid;
					
					if (bloodUnitGuid != Guid.Empty && patientGuid != Guid.Empty)
					{
						//CR 3015
						if (dtBCEMessages == null)
						{
							dtBCEMessages = OrderedUnit.UpdateBceForRelease(bloodUnitGuid, patientGuid, false);
						}
						else
						{
							dtTempBCEMessages = OrderedUnit.UpdateBceForRelease(bloodUnitGuid, patientGuid, false);
						    if (dtTempBCEMessages.Rows.Count > 0)
						    {
                                //*** Fortify Justified Code ***
                                //dtBCEMessages can't be null since it is checked earler (see above). 
                                dtBCEMessages.Rows.Add(dtTempBCEMessages.Rows[0].ItemArray);
                                //*** Fortify Justified Code ***
                            }
                        }
					}
				}
			}

			if (dtBCEMessages != null)
				dsSave.Tables.Add(dtBCEMessages);
			//BCE_MESSAGE end
			
			bool result =  DAL.OrderedTest.SaveTestResults(dsSave);

			// updating status of the order in Vista
			//BR_56.05, BR_56.30
			foreach( DataTable dtTestResults in dsTestResults.Tables )
			{
				if( dtTestResults.TableName != TABLE.Patient.TableName &&
					dtTestResults.TableName != "ABIDProcess")
				{
					if( dtTestResults.Columns.IndexOf(TABLE.OrderedTest.CprsOrderNumber) != -1 )
					{
						if( dtTestResults.Rows[0].IsNull(TABLE.OrderedTest.CprsOrderNumber) == false)
						{
							dtOrderedTestForVista = DAL.OrderedTest.GetOrderedTestForVista(dtTestResults.Rows[0][TABLE.OrderedTest.CprsOrderNumber].ToString());

							if( dtOrderedTestForVista.Rows.Count != 0 )
							{
								if( dtUpdateOrderedTests.Select( TABLE.OrderedTest.OrderedTestGuid + " = '" + dtTestResults.Rows[0][TABLE.OrderedTest.OrderedTestGuid].ToString() + "'").Length > 0 )
								{
									if( dtUpdateOrderedTests.Select(TABLE.OrderedTest.OrderedTestGuid + " = '" + dtTestResults.Rows[0][TABLE.OrderedTest.OrderedTestGuid].ToString() + "'")[0][TABLE.OrderedTest.TaskStatusCode].ToString() == Convert.ToString(Common.Utility.GetTaskStatusCodeFromEnum(Common.TaskStatus.Completed)) )
									{

										dtOrderedTestForVista.Rows[0][TABLE.OrderedTest.TaskStatusCode] = System.Convert.ToString(Common.Utility.GetTaskStatusCodeFromEnum(Common.TaskStatus.Completed));
										dtOrderedTestForVista.Rows[0][TABLE.OrderedTest.OrderStatusCode] = System.Convert.ToString(Common.Utility.GetOrderStatusCodeFromEnum(Common.OrderStatus.Complete));
				
										new VbecsOmgMessage().PatientOrderUpdate( dtOrderedTestForVista );
									}
								}
							}
						}
					}
				}
			}
			return result;
		}

		/// <summary>
		/// Initialize RackLot table
		/// </summary>
		/// <returns></returns>
		private static DataTable InitializeRackLotTable()
		{
			DataTable dtRackLots = new DataTable(TABLE.RackLot.TableName);

			dtRackLots.Columns.Add(TABLE.RackLot.RackLotGuid, typeof(Guid));
			dtRackLots.Columns.Add(TABLE.RackLot.RackGuid, typeof(Guid));
			dtRackLots.Columns.Add(TABLE.RackLot.ReagentGuid, typeof(Guid));
			dtRackLots.Columns.Add(TABLE.RackLot.TestWithName);
			dtRackLots.Columns.Add(TABLE.RackLot.TestWithId, typeof(int));
			dtRackLots.Columns.Add(TABLE.RackLot.LotVialNumber);
			dtRackLots.Columns.Add(TABLE.RackLot.ExpectedMinimumReactions);
			dtRackLots.Columns.Add(TABLE.RackLot.SatisfactoryResultIndicator, typeof(bool));
			dtRackLots.Columns.Add(TABLE.RackLot.ReactivityPhaseCode);
			dtRackLots.Columns.Add(TABLE.RackLot.LotUseDate, typeof(DateTime));
			dtRackLots.Columns.Add(TABLE.RackLot.DivisionCode);
			dtRackLots.Columns.Add(TABLE.RackLot.LastUpdateUser);
			dtRackLots.Columns.Add(TABLE.RackLot.ControlCellId, typeof(bool));
			dtRackLots.Columns.Add(TABLE.RackLot.Comments);
			dtRackLots.Columns.Add(TABLE.RackLot.RowVersion, typeof(byte[]));

			return dtRackLots;
		}

		/// <summary>
		/// Initialize RackTest table
		/// </summary>
		/// <returns></returns>
		private static DataTable InitializeRackTestTable(bool update)
		{
			DataTable dtRackTests = null;

			if (update == false)
			{
				dtRackTests = new DataTable("RackTestInsert");
				dtRackTests.Columns.Add(TABLE.RackTest.RackLotGuid, typeof(Guid));
				dtRackTests.Columns.Add(TABLE.RackTest.RackTestResultTypeCode);
			}
			else
			{
				dtRackTests = new DataTable("RackTestUpdate");
				dtRackTests.Columns.Add(TABLE.RackTest.RowVersion, typeof(byte[]));
			}

			dtRackTests.Columns.Add(TABLE.RackTest.DivisionCode);
			dtRackTests.Columns.Add(TABLE.RackTest.RackTestGuid, typeof(Guid));
			dtRackTests.Columns.Add(TABLE.RackTest.TestDate);
			dtRackTests.Columns.Add(TABLE.RackTest.TestTechId);
			dtRackTests.Columns.Add(TABLE.RackTest.RackTestResult);
			dtRackTests.Columns.Add(TABLE.RackTest.RecordStatusCode);
			dtRackTests.Columns.Add(TABLE.RackTest.LastUpdateUser);

			return dtRackTests;
		}

		private static DataTable InitializeAntiseraOrderedTestTable()
		{
			DataTable dtAntiseraOrderedTests = new DataTable(TABLE.AntiseraOrderedTest.TableName);
			dtAntiseraOrderedTests.Columns.Add(TABLE.AntiseraOrderedTest.AntiseraOrderedTestGuid);
			dtAntiseraOrderedTests.Columns.Add(TABLE.AntiseraOrderedTest.RackLotGuid);
			dtAntiseraOrderedTests.Columns.Add(TABLE.AntiseraOrderedTest.OrderedTestGuid);
			dtAntiseraOrderedTests.Columns.Add(TABLE.AntiseraOrderedTest.LastUpdateUser);
			dtAntiseraOrderedTests.Columns.Add(TABLE.AntiseraOrderedTest.RecordStatusCode);
	
			return dtAntiseraOrderedTests;
		}

		/// <summary>
		/// Initialize OrderedTest table for update
		/// </summary>
		private static DataTable InitializeOrderedTestForUpdateTable()
		{
			DataTable dtUpdateOrderedTests = new DataTable(TABLE.OrderedTest.TableName);
			dtUpdateOrderedTests.Columns.Add(TABLE.OrderedTest.OrderedTestGuid);
			dtUpdateOrderedTests.Columns.Add(TABLE.OrderedTest.TaskStatusCode);
			dtUpdateOrderedTests.Columns.Add(TABLE.OrderedTest.OrderStatusCode);
			dtUpdateOrderedTests.Columns.Add(TABLE.OrderedTest.LastUpdateUser);
			dtUpdateOrderedTests.Columns.Add(TABLE.OrderedTest.RowVersion, typeof(byte[]));
	
			return dtUpdateOrderedTests;
		}

		/// <summary>
		/// Initialize OrderedUnit table for update
		/// </summary>
		/// <returns></returns>
		private static DataTable InitializeOrderedUnitForUpdateTable()
		{
			DataTable dtUpdateOrderedUnits = new DataTable(TABLE.OrderedUnit.TableName);
			dtUpdateOrderedUnits.Columns.Add(TABLE.OrderedUnit.OrderedUnitGuid);
			dtUpdateOrderedUnits.Columns.Add(TABLE.OrderedUnit.CrossmatchDate);
			dtUpdateOrderedUnits.Columns.Add(TABLE.OrderedUnit.CrossmatchTechId);
			dtUpdateOrderedUnits.Columns.Add(TABLE.OrderedUnit.CrossmatchResultCode);
			dtUpdateOrderedUnits.Columns.Add(TABLE.OrderedUnit.LastUpdateUser);
			dtUpdateOrderedUnits.Columns.Add(TABLE.OrderedUnit.RecordStatusCode, typeof(char));
			dtUpdateOrderedUnits.Columns.Add(TABLE.OrderedUnit.ReleaseTechId);
			dtUpdateOrderedUnits.Columns.Add(TABLE.OrderedUnit.ReleaseDate, typeof(DateTime));
			dtUpdateOrderedUnits.Columns.Add(TABLE.OrderedUnit.RowVersion, typeof(byte[]));
			dtUpdateOrderedUnits.Columns.Add(TABLE.OrderedUnit.DivisionCode, typeof(string));

			return dtUpdateOrderedUnits;
		}

		///<Developers>
		///	<Developer>Margaret Jablonski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/10/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2994"> 
		///		<ExpectedInput>none</ExpectedInput>
		///		<ExpectedOutput>An empty data table</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="2763"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Initializes an empty OrderedTest data table
		/// </summary>
		/// <returns></returns>
		public static DataTable GetEmptyOrderedTestTableSchema(bool isUpdate)

		{
			DataTable dtOrderedTest = new DataTable(TABLE.OrderedTest.TableName);
			
			dtOrderedTest.Columns.Add(TABLE.OrderedTest.OrderedTestGuid, typeof(Guid));
			dtOrderedTest.Columns.Add(TABLE.OrderedTest.PatientOrderGuid, typeof(Guid));
			dtOrderedTest.Columns.Add(TABLE.OrderedTest.CprsOrderNumber, typeof(string));
			dtOrderedTest.Columns.Add(TABLE.OrderedTest.VbecsOrderNumber, typeof(int));
			dtOrderedTest.Columns.Add(TABLE.OrderedTest.RelatedComponentGuid, typeof(Guid));
			dtOrderedTest.Columns.Add(TABLE.OrderedTest.SpecimenRequiredIndicator, typeof(byte));			
			dtOrderedTest.Columns.Add(TABLE.OrderedTest.PatientSpecimenGuid, typeof(Guid));			
			dtOrderedTest.Columns.Add(TABLE.OrderedTest.OrderStatusCode, typeof(string));			
			dtOrderedTest.Columns.Add(TABLE.OrderedTest.OrderUrgencyCode, typeof(string));			
			dtOrderedTest.Columns.Add(TABLE.OrderedTest.OrderedTestComments, typeof(string));			
			dtOrderedTest.Columns.Add(TABLE.OrderedTest.RequiredResultsDate, typeof(DateTime));
			dtOrderedTest.Columns.Add(TABLE.OrderedTest.OrderableTestId, typeof(int));
			dtOrderedTest.Columns.Add(TABLE.OrderedTest.TestInterpretation, typeof(string)); 
			dtOrderedTest.Columns.Add(TABLE.OrderedTest.TestInterpretationComments, typeof(string));
			dtOrderedTest.Columns.Add(TABLE.OrderedTest.AutoInstrumentName, typeof(string));
			dtOrderedTest.Columns.Add(TABLE.OrderedTest.TaskStatusCode, typeof(string)); 
			dtOrderedTest.Columns.Add(TABLE.OrderedTest.RelatedTestGuid, typeof(Guid));
			dtOrderedTest.Columns.Add(TABLE.OrderedTest.CancellationCannedCommentGuid, typeof(Guid));
			dtOrderedTest.Columns.Add(TABLE.OrderedTest.CancellationText, typeof(string));
			dtOrderedTest.Columns.Add(TABLE.OrderedTest.DivisionCode, typeof(string));
			dtOrderedTest.Columns.Add(TABLE.OrderedTest.LabTestId, typeof(string));
			dtOrderedTest.Columns.Add(TABLE.OrderedTest.SpecimenAccessionNumber, typeof(string));
			dtOrderedTest.Columns.Add(TABLE.OrderedTest.SpecimenWorkloadUid, typeof(string));
			dtOrderedTest.Columns.Add(TABLE.OrderedTest.LastUpdateUser, typeof(string));
			
			if (isUpdate)
				dtOrderedTest.Columns.Add(TABLE.OrderedTest.RowVersion, typeof(byte[]));

			return dtOrderedTest;
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/10/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3009"> 
		///		<ExpectedInput>Exception Report, UpdateFunction</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="2764"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Inactivates test results for ordered test
		/// </summary>
		/// <param name="exRep">exception</param>
		/// <param name="lastUpdateFunctionId">UC or calling method</param>
		/// <returns></returns>
		public bool InactivateTestResults(BOL.ExceptionReport exRep, Common.UpdateFunction lastUpdateFunctionId)
		{
			DataTable dtInvalidatedTestException = exRep.ExceptionData.Table.Clone();
			dtInvalidatedTestException.Rows.Add(exRep.ExceptionData.ItemArray);

			DataTable dtOrderedTest = new DataTable(TABLE.OrderedTest.TableName);
			dtOrderedTest.Columns.Add(TABLE.OrderedTest.OrderedTestGuid, typeof(Guid));
			dtOrderedTest.Columns.Add(TABLE.OrderedTest.OrderableTestId, typeof(int));
			dtOrderedTest.Columns.Add(TABLE.SpecimenTest.InactivateDate, typeof(DateTime));
			dtOrderedTest.Columns.Add(TABLE.SpecimenTest.InactivateUser);
			dtOrderedTest.Columns.Add(TABLE.OrderedTest.LastUpdateUser);
			dtOrderedTest.Columns.Add(TABLE.OrderedTest.DivisionCode);
			dtOrderedTest.Columns.Add(TABLE.OrderedTest.LastUpdateFunctionId, typeof(int));

			DataRow drOrderedTest = dtOrderedTest.NewRow();
			drOrderedTest[TABLE.OrderedTest.OrderedTestGuid] = this.OrderedTestGuid;
			drOrderedTest[TABLE.OrderedTest.OrderableTestId] = -1;
			drOrderedTest[TABLE.SpecimenTest.InactivateUser] = Common.LogonUser.LogonUserName;
			drOrderedTest[TABLE.SpecimenTest.InactivateDate] = VBECSDateTime.GetDivisionCurrentDateTime();
			drOrderedTest[TABLE.OrderedTest.LastUpdateUser] = Common.LogonUser.LogonUserName;
			drOrderedTest[TABLE.OrderedTest.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
			drOrderedTest[TABLE.OrderedTest.LastUpdateFunctionId] = lastUpdateFunctionId;

			dtOrderedTest.Rows.Add(drOrderedTest);

			bool result = DAL.OrderedTest.InactivateTestResults(dtOrderedTest, dtInvalidatedTestException);
	
			if (result == true)
			{
				DataTable dt = DAL.OrderedTest.GetOrderedTest(this.OrderedTestGuid);
				if (dt.Rows.Count > 0)
					this.RowVersion = (byte[])dt.Rows[0][TABLE.OrderedTest.RowVersion];
			}

			return result;
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/10/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2767"> 
		///		<ExpectedInput>Exception Report, UpdateFunction, testid</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="2778"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Inactivates test results for ordered test and certain orderable test type
		/// </summary>
		/// <param name="exRep">ID</param>
		/// <param name="orderableTestId">ID</param>
		/// <param name="lastUpdateFunctionId">UC or calling method</param>
		/// <returns></returns>
		public bool InactivateTestResultsForOrderableTest(BOL.ExceptionReport exRep, int orderableTestId, Common.UpdateFunction lastUpdateFunctionId)
		{
			DataTable dtInvalidatedTestException = exRep.ExceptionData.Table.Clone();
			dtInvalidatedTestException.Rows.Add(exRep.ExceptionData.ItemArray);

			DataTable dtOrderedTest = new DataTable(TABLE.OrderedTest.TableName);
			dtOrderedTest.Columns.Add(TABLE.OrderedTest.OrderedTestGuid, typeof(Guid));
			dtOrderedTest.Columns.Add(TABLE.OrderedTest.OrderableTestId, typeof(int));
			dtOrderedTest.Columns.Add(TABLE.SpecimenTest.InactivateDate, typeof(DateTime));
			dtOrderedTest.Columns.Add(TABLE.SpecimenTest.InactivateUser);
			dtOrderedTest.Columns.Add(TABLE.OrderedTest.LastUpdateUser);
			dtOrderedTest.Columns.Add(TABLE.OrderedTest.DivisionCode);
			dtOrderedTest.Columns.Add(TABLE.OrderedTest.LastUpdateFunctionId, typeof(int));

			DataRow drOrderedTest = dtOrderedTest.NewRow();
			drOrderedTest[TABLE.OrderedTest.OrderedTestGuid] = this.OrderedTestGuid;
			drOrderedTest[TABLE.OrderedTest.OrderableTestId] = orderableTestId;
			drOrderedTest[TABLE.SpecimenTest.InactivateUser] = Common.LogonUser.LogonUserName;
			drOrderedTest[TABLE.SpecimenTest.InactivateDate] = VBECSDateTime.GetDivisionCurrentDateTime();
			drOrderedTest[TABLE.OrderedTest.LastUpdateUser] = Common.LogonUser.LogonUserName;
			drOrderedTest[TABLE.OrderedTest.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
			drOrderedTest[TABLE.OrderedTest.LastUpdateFunctionId] = lastUpdateFunctionId;

			dtOrderedTest.Rows.Add(drOrderedTest);

			bool result = DAL.OrderedTest.InactivateTestResults(dtOrderedTest, dtInvalidatedTestException);

			if (result == true)
			{
				DataTable dt = DAL.OrderedTest.GetOrderedTest(this.OrderedTestGuid);
				if (dt.Rows.Count > 0)
					this.RowVersion = (byte[])dt.Rows[0][TABLE.OrderedTest.RowVersion];
			}

			return result;
		}

		/// <summary>
		/// Loads properties from DataRow
		/// </summary>
		/// <param name="dtRow"></param>
		protected new void LoadFromDataRow(DataRow dtRow)
		{
			if (dtRow.Table.Columns.Contains(TABLE.OrderedTest.SpecimenAccessionNumber))
			{
				this.SpecimenAccessionNumber = dtRow.IsNull(TABLE.OrderedTest.SpecimenAccessionNumber) ? String.Empty : dtRow[TABLE.OrderedTest.SpecimenAccessionNumber].ToString();
			}
			if (dtRow.Table.Columns.Contains(TABLE.OrderedTest.VbecsOrderNumber))
			{
				this.VBECSOrderNumber = dtRow.IsNull(TABLE.OrderedTest.VbecsOrderNumber) ? String.Empty : dtRow[TABLE.OrderedTest.VbecsOrderNumber].ToString();
			}
			if (dtRow.Table.Columns.Contains(TABLE.OrderedTest.OrderedTestGuid))
			{
				this.OrderedTestGuid = dtRow.IsNull(TABLE.OrderedTest.OrderedTestGuid) ? Guid.NewGuid() : (System.Guid) dtRow[TABLE.OrderedTest.OrderedTestGuid];
			}
			if (dtRow.Table.Columns.Contains(TABLE.OrderedTest.CprsOrderNumber))
			{
				this.CPRSOrderNumber = dtRow.IsNull(TABLE.OrderedTest.CprsOrderNumber) ? String.Empty : dtRow[TABLE.OrderedTest.CprsOrderNumber].ToString();
			}
			if (dtRow.Table.Columns.Contains(TABLE.OrderedTest.OrderUrgencyCode))
			{
				this.OrderUrgencyCode = dtRow.IsNull(TABLE.OrderedTest.OrderUrgencyCode) ? String.Empty : dtRow[TABLE.OrderedTest.OrderUrgencyCode].ToString();
			}
			if (dtRow.Table.Columns.Contains(TABLE.OrderUrgency.OrderUrgencyText))
			{
				this.OrderUrgencyText = dtRow.IsNull(TABLE.OrderUrgency.OrderUrgencyText) ? String.Empty : dtRow[TABLE.OrderUrgency.OrderUrgencyText].ToString();
			}
			if (dtRow.Table.Columns.Contains(TABLE.OrderedTest.OrderedTestComments))
			{
				this.Comments = dtRow.IsNull(TABLE.OrderedTest.OrderedTestComments) ? String.Empty : dtRow[TABLE.OrderedTest.OrderedTestComments].ToString();
			}
			if (dtRow.Table.Columns.Contains(TABLE.OrderedTest.RequiredResultsDate))
			{
				this.RequiredResultsDate = dtRow.IsNull(TABLE.OrderedTest.RequiredResultsDate) ? DateTime.MinValue : (DateTime) dtRow[TABLE.OrderedTest.RequiredResultsDate];
			}
			if (dtRow.Table.Columns.Contains(TABLE.OrderedTest.OrderableTestId))
			{
				this.OrderableTestId = dtRow.IsNull(TABLE.OrderedTest.OrderableTestId) ? int.MinValue : (int) dtRow[TABLE.OrderedTest.OrderableTestId];
			}
			if (dtRow.Table.Columns.Contains(TABLE.OrderedTest.OrderStatusCode))
			{
				this.OrderStatusCode = dtRow.IsNull(TABLE.OrderedTest.OrderStatusCode) ? Common.OrderStatus.Unknown : Common.Utility.GetOrderStatusCodeFromString(dtRow[TABLE.OrderedTest.OrderStatusCode].ToString());
			}
			if (dtRow.Table.Columns.Contains(TABLE.OrderStatus.OrderStatusText))
			{
				this.OrderStatusText = dtRow.IsNull(TABLE.OrderStatus.OrderStatusText) ? String.Empty : dtRow[TABLE.OrderStatus.OrderStatusText].ToString();
			}
			if (dtRow.Table.Columns.Contains(TABLE.OrderableTest.OrderableTestName))
			{
				this.OrderableTest = dtRow.IsNull(TABLE.OrderableTest.OrderableTestName) ? String.Empty : dtRow[TABLE.OrderableTest.OrderableTestName].ToString();
			}
			if (dtRow.Table.Columns.Contains(TABLE.BloodUnit.RecordStatusCode))
			{
				this.RecordStatusCode =	dtRow.IsNull(TABLE.BloodUnit.RecordStatusCode) ? Common.RecordStatusCode.Unknown : Common.Utility.GetRecordStatusCodeFromString(dtRow[TABLE.BloodUnit.RecordStatusCode].ToString());
			}
			if (dtRow.Table.Columns.Contains(TABLE.OrderedTest.TaskStatusCode))
			{
				this.TaskStatusCode =	dtRow.IsNull(TABLE.OrderedTest.TaskStatusCode) ? char.MinValue : System.Convert.ToChar(dtRow[TABLE.OrderedTest.TaskStatusCode]);
			}
			if (dtRow.Table.Columns.Contains(TABLE.TaskStatus.TaskStatusText))
			{
				this.TaskStatusText =	dtRow.IsNull(TABLE.TaskStatus.TaskStatusText) ? string.Empty : dtRow[TABLE.TaskStatus.TaskStatusText].ToString();
			}
			if (dtRow.Table.Columns.Contains(TABLE.SpecimenStatus.SpecimenStatusText))
			{
				this.SpecimenStatusText = dtRow.IsNull(TABLE.SpecimenStatus.SpecimenStatusText) ? String.Empty : dtRow[TABLE.SpecimenStatus.SpecimenStatusText].ToString();
			}
			if (dtRow.Table.Columns.Contains(TABLE.OrderedTest.PatientSpecimenGuid))
			{
				this.Specimen = dtRow.IsNull(TABLE.OrderedTest.PatientSpecimenGuid) ? new Specimen() : new Specimen( (Guid) dtRow[TABLE.OrderedTest.PatientSpecimenGuid] );
			}
			if (dtRow.Table.Columns.Contains(TABLE.Patient.PatientGuid))
			{
				this.Patient = dtRow.IsNull(TABLE.Patient.PatientGuid) ? new Patient() : new Patient( (Guid) dtRow[TABLE.Patient.PatientGuid] );
			}
			if (dtRow.Table.Columns.Contains(TABLE.OrderedTest.RequiredResultsDate))
			{
				this.RequiredResultsDate = dtRow.IsNull(TABLE.OrderedTest.RequiredResultsDate) ? DateTime.MinValue : (DateTime)dtRow[TABLE.OrderedTest.RequiredResultsDate];
			}
			if (dtRow.Table.Columns.Contains(TABLE.OrderedTest.RowVersion))
			{
				this.RowVersion	= dtRow.IsNull(TABLE.OrderedTest.RowVersion) ? new byte[0] : (byte[]) dtRow[TABLE.OrderedTest.RowVersion];
			}
			if (dtRow.Table.Columns.Contains(TABLE.OrderedTest.RelatedComponentGuid))
			{
				this.RelatedComponentGuid = dtRow.IsNull(TABLE.OrderedTest.RelatedComponentGuid) ? Guid.Empty : (Guid)dtRow[TABLE.OrderedTest.RelatedComponentGuid];
			}
			if (dtRow.Table.Columns.Contains(TABLE.OrderedTest.RelatedTestGuid))
			{
				this.RelatedTestGuid = dtRow.IsNull(TABLE.OrderedTest.RelatedTestGuid) ? Guid.Empty : (Guid)dtRow[TABLE.OrderedTest.RelatedTestGuid];
			}
			if (dtRow.Table.Columns.Contains(TABLE.OrderedTest.LabOrderNumber))
			{
				_labOrderNumber = dtRow.IsNull(TABLE.OrderedTest.LabOrderNumber) ? string.Empty : dtRow[TABLE.OrderedTest.LabOrderNumber].ToString();
			}
			if (dtRow.Table.Columns.Contains(TABLE.OrderedTest.LabTestId))
			{
				this.LabTestId = dtRow.IsNull(TABLE.OrderedComponent.LabTestId) ? string.Empty : dtRow[TABLE.OrderedComponent.LabTestId].ToString();
			}
			if (dtRow.Table.Columns.Contains(TABLE.BloodTestType.ReportableTestId))
			{
				this.ReportableTestId = dtRow.IsNull(TABLE.BloodTestType.ReportableTestId) ? int.MinValue : (int)dtRow[TABLE.BloodTestType.ReportableTestId];
			}
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/18/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3261"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3262"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns a data table of test orders
		/// </summary>
		/// <returns></returns>
		public static DataTable GetTestOrders()
		{
			return(DAL.OrderedTest.GetTestOrders());
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/18/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3263"> 
		///		<ExpectedInput>Valid parameters</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3264"> 
		///		<ExpectedInput>Invalid parameters</ExpectedInput>
		///		<ExpectedOutput>DataAccessLayerException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Updates an ordered test's status
		/// </summary>
		/// <param name="orderStatusCode"></param>
		/// <param name="taskStatusCode"></param>
		/// <param name="userName"></param>
		/// <param name="cannedCommentGuid"></param>
		/// <param name="cannedCommentText"></param>
		/// <param name="cprsOrderNumber"></param>
		/// <param name="dtSpecimen"></param>
		/// <param name="lastUpdateFunctionId">UC or calling method</param>
		/// <returns></returns>
		public bool UpdateStatus(char orderStatusCode, 
			char taskStatusCode, 
			string userName, 
			Guid cannedCommentGuid, 
			string cannedCommentText, 
			string cprsOrderNumber, 
			DataTable dtSpecimen, 
			//DataTable dtSpecimenComment, 
			Common.UpdateFunction lastUpdateFunctionId)
		{
			//Create new Order Test table 
			DataTable dtOrderTest = GetEmptyOrderedTestTableSchema(true);
			DataTable dtWorkloadEvents = null;

			//Create a new Row in Order Test table and add data based on the information passed in
			DataRow drOrderTest = dtOrderTest.NewRow();

			drOrderTest[TABLE.OrderedTest.OrderedTestGuid] = this.OrderedTestGuid;

			drOrderTest[TABLE.OrderedTest.SpecimenWorkloadUid] = this.Specimen.SpecimenUid;
			drOrderTest[TABLE.OrderedTest.SpecimenAccessionNumber] = this.SpecimenAccessionNumber;
			if (orderStatusCode != Common.Utility.GetOrderStatusCodeFromEnum(Common.OrderStatus.Cancelled))
			{
				drOrderTest[TABLE.OrderedTest.PatientSpecimenGuid] = this.Specimen.SpecimenGuid;
			}
			else
			{
				drOrderTest[TABLE.OrderedTest.PatientSpecimenGuid] =  System.DBNull.Value;
			}
			drOrderTest[TABLE.OrderedTest.OrderStatusCode] = orderStatusCode;
			if (taskStatusCode == char.MinValue) 
			{
				drOrderTest[TABLE.OrderedTest.TaskStatusCode] = System.DBNull.Value;
			}
			else
			{
				drOrderTest[TABLE.OrderedTest.TaskStatusCode] = taskStatusCode;
			}
			drOrderTest[TABLE.OrderedTest.LastUpdateUser] = userName;
			if (cannedCommentGuid == Guid.Empty)
			{
				drOrderTest[TABLE.OrderedTest.CancellationCannedCommentGuid] = System.DBNull.Value;
			}
			else
			{
				drOrderTest[TABLE.OrderedTest.CancellationCannedCommentGuid] = cannedCommentGuid;
			}
			drOrderTest[TABLE.OrderedTest.CancellationText] = cannedCommentText;
			drOrderTest[TABLE.OrderedTest.RowVersion] = this.RowVersion;

			//Add Order Test row to the Order Test table
			dtOrderTest.Rows.Add(drOrderTest);
		
			if (cprsOrderNumber.Trim() != string.Empty)
			{
				if (UpdateCPRS(cprsOrderNumber, orderStatusCode, taskStatusCode, cannedCommentGuid, cannedCommentText) == false)
					return false;
			}

			System.Guid [] orderedTestGuid = new System.Guid[1];
			orderedTestGuid[0] = this.OrderedTestGuid;
			if(orderStatusCode == 'X')
			{				
				dtWorkloadEvents = BOL.WorkloadEvent.GenerateWorkloadData(Common.WorkloadProcessID.CancelOrder, this.OrderedTestGuid, System.Guid.Empty, orderedTestGuid, Common.WorkloadTransactionType.Miscellanious, false);
			}
			else if(orderStatusCode == 'A')
			{
				dtWorkloadEvents = BOL.WorkloadEvent.GenerateWorkloadData(Common.WorkloadProcessID.AcceptOrder, this.OrderedTestGuid, System.Guid.Empty, orderedTestGuid, Common.WorkloadTransactionType.Miscellanious, false);
			}
			

			return DAL.OrderedTest.UpdateStatus(
				this.OrderedTestGuid,
				orderStatusCode,
				taskStatusCode,
				userName,
				cannedCommentGuid,
				cannedCommentText,
				cprsOrderNumber,
				dtSpecimen,
				dtOrderTest,
				dtWorkloadEvents,
				lastUpdateFunctionId
				);
		}

		private bool UpdateCPRS(string cprsOrderNumber,
			char orderStatusCode,
			char taskStatusCode,
			Guid cannedCommentGuid,
			string cannedCommentText)
		{
			DataSet ds = DAL.OrderedTest.GetOrderData(cprsOrderNumber);			

			if (ds.Tables.Count == 1  && ds.Tables[0].Rows.Count == 1)
			{
				DataTable dtOrderData = ds.Tables[0];
			
				dtOrderData.Rows[0][TABLE.OrderedTest.OrderStatusCode] = orderStatusCode;
				dtOrderData.Rows[0][TABLE.OrderedTest.TaskStatusCode] = taskStatusCode;
				if (cannedCommentGuid == Guid.Empty)
				{
					dtOrderData.Rows[0][TABLE.OrderedTest.CancellationCannedCommentGuid] = System.DBNull.Value;
				}
				else
				{
					dtOrderData.Rows[0][TABLE.OrderedTest.CancellationCannedCommentGuid] = cannedCommentGuid;
				}
				dtOrderData.Rows[0][TABLE.CannedComment.CannedCommentText] = cannedCommentText;
		
				new VbecsOmgMessage().PatientOrderUpdate(dtOrderData);
				return true;
			}
			else
			{
				throw new BOL.BusinessObjectException("Data error: More than 1 row returned");
			}
		}
 
		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/10/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3018"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>The valid ReagentType object</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="2795"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns a ReagentType for an orderable test
		/// </summary>
		/// <returns></returns>
		public ReagentType GetReagentTypeForOrderableTest()
		{
			DataTable dt = DAL.OrderedTest.GetReagentTypeForOrderableTest(this.OrderableTestId);

			if (dt.Rows.Count == 0)
			{
				return null;
			}
			else
			{
				return new ReagentType(dt.Rows[0]);
			}
		}
	

		///<Developers>
		///	<Developer>Margaret Jablonski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/10/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2995"> 
		///		<ExpectedInput>List of selected tests, datatable of selected units, valid Patient GUID
		///						valid Division Code, valid Patient Specimen GUID, valid Parent Test GUID,
		///						valid Parent Component GUID, valid Canned Comment ID and Canned Comment Text 
		///		</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2997"> 
		///		<ExpectedInput>Empty list of selected tests, no datatable of selected units, invalid Patient GUID
		///						invalid Division Code, valid Patient Specimen GUID, valid Parent Test GUID,
		///						invalid Parent Component GUID, invalid Canned Comment ID and Canned Comment Text 
		///						</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Save Order Reflex tests data from UC_45. Saves data to up to four tables
		/// Implements BR_45.05
		/// </summary>
		/// <param name="selectedTestsList"></param>
		/// <param name="dtSelectedUnits"></param>
		/// <param name="specimen"></param>
		/// <param name="patientTreatmentGuid"></param>
		/// <param name="parentTestGuid"></param>
		/// <param name="parentComponentGuid"></param>
		/// <param name="expiredCommentText"></param>
		/// <param name="lastUpdateFunctionId">UC or calling method</param>
		/// <returns></returns>
		public static bool SaveReflexTests(int[] selectedTestsList, DataTable dtSelectedUnits, 
			BOL.Specimen specimen, Guid patientTreatmentGuid,
			Guid parentTestGuid, Guid parentComponentGuid, 
			string expiredCommentText, Common.UpdateFunction lastUpdateFunctionId)
		{
			DataSet dsSave = new DataSet();
			System.DateTime currentDate;
			DataTable dtOrderableTests = BOL.OrderedTest.GetOrderableTestList();
			DataTable dtExceptionsReflexTest = null;
			//CR 1843
			Guid patientOrderGuid = Guid.Empty;

			currentDate = BOL.VBECSDateTime.GetDivisionCurrentDateTime();
			
			//Create new tables 
			DataTable dtPatientOrder = BOL.PatientOrder.InitializePatientOrderTable();
			DataTable dtOrderedTests = GetEmptyOrderedTestTableSchema(false);

			//Create a new Row in PatientOrder table and add data based on the information passed in
			DataRow drPatientOrder = dtPatientOrder.NewRow();

			VbecsUser user = new BOL.VbecsUser(Common.LogonUser.LogonUserName);

			drPatientOrder[TABLE.PatientOrder.PatientOrderGuid] = Guid.NewGuid();
			drPatientOrder[TABLE.PatientOrder.PatientTreatmentGuid] = patientTreatmentGuid;
			drPatientOrder[TABLE.PatientOrder.OrderPlacedDatetime] = currentDate;
			drPatientOrder[TABLE.PatientOrder.OrderReceivedDatetime] = currentDate;
			if (user != null)
			{
				if (user.UserName != null)
				{
					drPatientOrder[TABLE.PatientOrder.OrderEnteredByLastName] = user.UserName;
					drPatientOrder[TABLE.PatientOrder.OrderingProviderLastName] = user.UserName;
				}
			}
			drPatientOrder[TABLE.PatientOrder.SpecimenRequiredIndicator] = 1; //1 = Yes
			drPatientOrder[TABLE.PatientOrder.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
			drPatientOrder[TABLE.PatientOrder.LastUpdateUser] = Common.LogonUser.LogonUserName;

			//CR 1843 adding OrderGroupNumber to the PatientOrder table
			if(parentTestGuid == Guid.Empty && parentComponentGuid != Guid.Empty)
			{
				//Initial test was a component test
				DataTable dtComponentOrder = BOL.OrderedComponent.GetOrderedComponent(parentComponentGuid);
				if (dtComponentOrder.Rows.Count > 0)
				{
					patientOrderGuid = (Guid)dtComponentOrder.Rows[0][TABLE.OrderedComponent.PatientOrderGuid];
				}
			}
			else if(parentTestGuid != Guid.Empty && parentComponentGuid == Guid.Empty)
			{
				//Initial test was a diagnostic test
				DataTable dtTest = BOL.OrderedTest.GetOrderedTest(parentTestGuid);
				if (dtTest.Rows.Count > 0)
				{
					patientOrderGuid = (Guid)dtTest.Rows[0][TABLE.OrderedTest.PatientOrderGuid];
				}
			}
			if (patientOrderGuid != Guid.Empty)
			{
				DataTable dtParentOrder = DAL.PatientOrder.GetPatientOrder(patientOrderGuid);
				if (dtParentOrder.Rows.Count > 0)
				{
					string orderGroupNumber = dtParentOrder.Rows[0][TABLE.PatientOrder.OrderGroupNumber].ToString();
					drPatientOrder[TABLE.PatientOrder.OrderGroupNumber] = orderGroupNumber;
				}
			}

			//Add PatientOrder row to the PatientOrder table
			dtPatientOrder.Rows.Add(drPatientOrder);
			
			//Add PatientOrder table to dsSave dataset
			dsSave.Tables.Add(Common.Utility.AppendLastUpdateInformation(dtPatientOrder, lastUpdateFunctionId));
			
			foreach(int selectedTest in selectedTestsList)
			{
				//Create new Rows in dtOrderedTests table and add data based on the information passed in
				DataRow drOrderedTest = dtOrderedTests.NewRow();
				
				//Create new VBECS order number
				DataTable dtVbecsOrderNumber = DAL.PatientOrder.GetNextVbecsOrderNumber();

				drOrderedTest[TABLE.OrderedTest.OrderedTestGuid] = Guid.NewGuid();
				drOrderedTest[TABLE.OrderedTest.PatientOrderGuid] = drPatientOrder[TABLE.OrderedTest.PatientOrderGuid];
				drOrderedTest[TABLE.OrderedTest.VbecsOrderNumber] = dtVbecsOrderNumber.Rows[0][TABLE.OrderedTest.VbecsOrderNumber]; 
				
				if(parentTestGuid == Guid.Empty && parentComponentGuid != Guid.Empty)
				{
					//Initial test was a component test
					drOrderedTest[TABLE.OrderedTest.RelatedComponentGuid] = parentComponentGuid;  
				}
				else if(parentTestGuid != Guid.Empty && parentComponentGuid == Guid.Empty)
				{
					//Initial test was a diagnostic test
					drOrderedTest[TABLE.OrderedTest.RelatedTestGuid] = parentTestGuid;
				}

				drOrderedTest[TABLE.OrderedTest.SpecimenRequiredIndicator] = 1; //1 = Yes
				drOrderedTest[TABLE.OrderedTest.PatientSpecimenGuid] = specimen.SpecimenGuid;
				drOrderedTest[TABLE.OrderedTest.OrderStatusCode] = Common.Utility.GetOrderStatusCodeFromEnum(Common.OrderStatus.Accepted); 
				drOrderedTest[TABLE.OrderedTest.OrderUrgencyCode] = Common.Utility.GetOrderUrgencyCodeFromEnum(Common.OrderUrgency.Routine); //Routine - BR_45.05 
				drOrderedTest[TABLE.OrderedTest.OrderableTestId] = selectedTest;
				drOrderedTest[TABLE.OrderedTest.TaskStatusCode] = Common.Utility.GetTaskStatusCodeFromEnum(Common.TaskStatus.NotStarted); 
				drOrderedTest[TABLE.OrderedTest.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
				drOrderedTest[TABLE.OrderedTest.LastUpdateUser] = Common.LogonUser.LogonUserName;

				//Add Ordered Test row to the Ordered Test table
				dtOrderedTests.Rows.Add(drOrderedTest);

				//generate exception if the specimen is expired
				if (expiredCommentText != null && expiredCommentText != string.Empty)
				{
					string ReflexOrderName = string.Empty;

					if (dtOrderableTests.Select(TABLE.OrderableTest.OrderableTestId + " = " + selectedTest).Length > 0)
					{
						ReflexOrderName = dtOrderableTests.Select(TABLE.OrderableTest.OrderableTestId + " = " + selectedTest)[0][TABLE.OrderableTest.OrderableTestName].ToString();
					}

					BOL.ExceptionReport exRep = new BOL.ExceptionReport(Common.ExceptionType.Reflex_test_ordered_on_expired_specimen, expiredCommentText, Common.LogonUser.LogonUserName); 
					exRep.GenerateExceptionReflexTest(specimen, ReflexOrderName);

					if (dtExceptionsReflexTest == null)
					{
						dtExceptionsReflexTest = exRep.ExceptionData.Table.Clone();
					}
                    //*** Fortify Justified Code ***
                    //dtExceptionsReflexTest can't be null since it is checked earler (see above). 
                    dtExceptionsReflexTest.Rows.Add(exRep.ExceptionData.ItemArray);
                    //*** Fortify Justified Code ***
                }
            }

			//Add dtOrderedTests table to dsSave dataset
			dsSave.Tables.Add(Common.Utility.AppendLastUpdateInformation(dtOrderedTests, lastUpdateFunctionId));

			if (dtExceptionsReflexTest != null)
				dsSave.Tables.Add(dtExceptionsReflexTest);

//			//No need to create specimen comment. It is going to exception report instead

//			//Add this table only if the test is ordered on the expired specimen
//			if (cannedCommentGuid != Guid.Empty)
//			{
//				//Create new tables 
//				DataTable dtPatientSpecimenComment = DAL.SpecimenComment.GetEmptyPatientSpecimenCommentTable(true);
//
//				//Create a new Row in dtPatientSpecimenComment table and add data based on the information passed in
//				DataRow drPSComment = dtPatientSpecimenComment.NewRow();
//
//				drPSComment[TABLE.PatientSpecimenComment.PatientSpecimenCommentGuid] = Guid.NewGuid();
//				drPSComment[TABLE.PatientSpecimenComment.PatientSpecimenGuid] = specimen.SpecimenGUID;
//				drPSComment[TABLE.PatientSpecimenComment.CannedCommentGuid] = cannedCommentGuid;
//				drPSComment[TABLE.PatientSpecimenComment.PatientSpecimenCommentText] = commentText;
//				drPSComment[TABLE.PatientSpecimenComment.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
//
//				//Add Patient Specimen Comment row to the dtPatientSpecimenComment table
//				dtPatientSpecimenComment.Rows.Add(drPSComment);
//
//				//Add dtPatientSpecimenComment table to dsSave dataset
//				dsSave.Tables.Add(Common.Utility.AppendLastUpdateInformation(dtPatientSpecimenComment, lastUpdateFunctionId));
//			}
			
			//Add this table only for the Repeat Crossmatch test
			if (dtSelectedUnits.Rows.Count > 0)
			{
				//Create new table
				DataTable dtOrderedUnits = BOL.OrderedUnit.InitializeOrderedUnitTable();
				
				foreach (DataRow drSelectedUnits in dtSelectedUnits.Rows)
				{
					//Create a new Rows in dtOrderedUnits table and add data based on the information passed in
					DataRow drOrderedUnit = dtOrderedUnits.NewRow();

					drOrderedUnit[TABLE.OrderedUnit.OrderedUnitGuid] = Guid.NewGuid();
					//Note: there should be only one row in dtOrderedTests for this case. 
					drOrderedUnit[TABLE.OrderedUnit.OrderedTestGuid] = dtOrderedTests.Rows[0][TABLE.OrderedTest.OrderedTestGuid];
					drOrderedUnit[TABLE.OrderedUnit.BloodUnitGuid] = drSelectedUnits[TABLE.OrderedUnit.BloodUnitGuid]; 
					drOrderedUnit[TABLE.OrderedUnit.SelectedDate] = drSelectedUnits[TABLE.OrderedUnit.SelectedDate];
					drOrderedUnit[TABLE.OrderedUnit.SelectedTechId] = drSelectedUnits[TABLE.OrderedUnit.SelectedTechId];
					drOrderedUnit[TABLE.OrderedUnit.CrossmatchRequiredIndicator] = true;
					drOrderedUnit[TABLE.OrderedUnit.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Active);
					drOrderedUnit[TABLE.OrderedUnit.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
					drOrderedUnit[TABLE.OrderedUnit.LastUpdateUser] = Common.LogonUser.LogonUserName;

					//Add OrderedUnits row to the Ordered Units table
					dtOrderedUnits.Rows.Add(drOrderedUnit);
				}

				//Add drOrderedUnit table to dsSave dataset
				dsSave.Tables.Add(Common.Utility.AppendLastUpdateInformation(dtOrderedUnits, lastUpdateFunctionId));
			}

			return DAL.OrderedTest.SaveReflexTests(dsSave);
            
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/10/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3020"> 
		///		<ExpectedInput>The valid OrderableTestId</ExpectedInput>
		///		<ExpectedOutput>true or false</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="2796"> 
		///		<ExpectedInput>invalid  OrderableTestId</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Check if test is cprs orderable
		/// </summary>
		/// <param name="orderableTestId"></param>
		/// <returns></returns>
		public static bool IsCprsOrderable(int orderableTestId)
		{
			int CprsOrderableItemId = DAL.OrderedTest.GetCprsOrderable(orderableTestId);

			if (CprsOrderableItemId == 99 ||
				CprsOrderableItemId == -1)
			{
				return false;
			}
			else
			{
				return true;
			}
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/22/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3056"> 
		///		<ExpectedInput>Guid</ExpectedInput>
		///		<ExpectedOutput>The DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3057"> 
		///		<ExpectedInput>Invalid Guid</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Get the test results for task
		/// </summary>
		/// <param name="orderedTestGuid"></param>
		/// <returns></returns>
		public static DataTable GetTestResultsForTask(Guid orderedTestGuid)
		{
			return DAL.OrderedTest.GetTestResultsForTask(orderedTestGuid);
		}

		///<Developers>
		///	<Developer></Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/11/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7516"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7517"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Get test results that were marked as erroneous for task 
		/// </summary>
		/// <param name="orderedTestGuid"></param>
		/// <returns></returns>
		public static DataTable GetTestResultsWithErrorForTask(Guid orderedTestGuid)
		{
			return DAL.OrderedTest.GetTestResultsWithErrorForTask(orderedTestGuid);
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/29/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3092"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>The DataTable</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="2797"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Get the historical test results for the ordered test. This method returns the list of test results
		/// for the same type of testing and the same patient as the original test
		/// </summary>
		/// <returns></returns>
		public DataTable GetHistoricalTestResultsForOrderedTest()
		{
			return DAL.OrderedTest.GetHistoricalTestResultsForOrderedTest(this.OrderedTestGuid);
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/29/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2798"> 
		///		<ExpectedInput>Little Date</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="2799"> 
		///		<ExpectedInput>Big Date</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Checks if the specimen is expired at the time of testing - BR_40.01
		/// </summary>
		/// <param name="testingDateTime"></param>
		/// <returns></returns>
		public bool CheckForExpiredSpecimen(DateTime testingDateTime)
		{
			if (this.Specimen.SpecimenExpirationDate < testingDateTime)
				return false;
			else
				return true;
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/29/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2800"> 
		///		<ExpectedInput>Negative ABS, Anti-C from VBECS</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="0" testid ="9146"> 
		///		<ExpectedInput>Negative ABS, Anti-C from conversion</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="9147"> 
		///		<ExpectedInput>Positive ABS, No TRs</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="9148"> 
		///		<ExpectedInput>Negative ABS, No TRs</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2801"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Checks if the test is eligible for IS only crossmatch - BR_40.05
		/// </summary>
		/// <returns></returns>
		public bool CheckIfISOnlyXMatchPossible()
		{
			//CR 2900 replaced the code for this method with code matching CheckIfISOnlyXMatchPossible from OrderedComponent class 
			bool PositiveABS = false;
			bool AntigenNegativeReqExists = false;
			string ABSInterpBloodTestType = "";
			string TASABSInterpBloodTestType = "";
			int ABSInterpCount = 0;

			DataTable dtTestResults = this.Specimen.GetTestResultsForSpecimen();
			DataTable dtTransfusionRequirements = BOL.PatientTransfusionRequirement.GetTransfusionRequirementsForPatient(this.Patient.PatientGuid);
			DataTable dtBloodTestTypes = BOL.BloodTestType.GetBloodTestTypes(((int)Common.OrderableTest.ABS).ToString());
			
			if (dtBloodTestTypes.Select(TABLE.BloodTestType.BloodTestName + " LIKE '*Interp*'").Length > 0)
			{
				ABSInterpBloodTestType = dtBloodTestTypes.Select(TABLE.BloodTestType.BloodTestName + " LIKE '*Interp*'")[0][TABLE.BloodTestType.BloodTestTypeId].ToString().Trim();
			}

			dtBloodTestTypes = BOL.BloodTestType.GetBloodTestTypes(((int)Common.OrderableTest.TAS).ToString());
			if (dtBloodTestTypes.Select(TABLE.BloodTestType.BloodTestName + " LIKE '*Antibody Screen Interp*'").Length > 0)
			{
				TASABSInterpBloodTestType = dtBloodTestTypes.Select(TABLE.BloodTestType.BloodTestName + " LIKE '*Antibody Screen Interp*'")[0][TABLE.BloodTestType.BloodTestTypeId].ToString().Trim();
			}

			//checking if there are any positive results of ABS testing
			foreach (DataRow drTestResult in dtTestResults.Rows)
			{
				if (drTestResult[TABLE.BloodTestType.BloodTestTypeId].ToString().Trim() == ABSInterpBloodTestType ||
					drTestResult[TABLE.BloodTestType.BloodTestTypeId].ToString().Trim() == TASABSInterpBloodTestType)
				{
					ABSInterpCount ++;
					if (drTestResult[TABLE.TestResult.TestResultId].ToString().Trim() != Common.TestResult.N.ToString())
					{
						PositiveABS = true;
					}
					else
					{
						PositiveABS = false;
					}
				}
			}

			if (ABSInterpCount == 0)
				PositiveABS = true;

			if (dtTransfusionRequirements.Select(TABLE.PatientTransfusionRequirement.TransfusionRequirementCategoryCode + " = '" + Common.Utility.GetTransfusionCategoryCodeFromEnum(Common.TransfusionRequirementCategoryCode.AntigenNegative) + "' AND " + TABLE.PatientTransfusionRequirement.RecordStatusCode + " IN ('A','C')").Length > 0)
			{
				AntigenNegativeReqExists = true;
			}
			else if (dtTransfusionRequirements.Select(TABLE.PatientTransfusionRequirement.TransfusionRequirementCategoryCode + " = '" + Common.Utility.GetTransfusionCategoryCodeFromEnum(Common.TransfusionRequirementCategoryCode.AntibodyIdentified) + "' AND " + TABLE.PatientTransfusionRequirement.RecordStatusCode + " IN ('A','C')").Length > 0)
			{
				DataTable dtAntibodies = BOL.AntibodyType.GetAntibodyTypeLookupList();
				DataRow[] drTRs = dtTransfusionRequirements.Select(TABLE.PatientTransfusionRequirement.TransfusionRequirementCategoryCode + " = '" + Common.Utility.GetTransfusionCategoryCodeFromEnum(Common.TransfusionRequirementCategoryCode.AntibodyIdentified) + "' AND " + TABLE.PatientTransfusionRequirement.RecordStatusCode + " IN ('A','C')");
				foreach(DataRow drTR in drTRs)
				{
					if (drTR.IsNull(TABLE.PatientTransfusionRequirement.AntibodyTypeId) == false)
					{
						if (dtAntibodies.Select(TABLE.AntibodyType.AntibodyTypeId + " = " + (int)drTR[TABLE.PatientTransfusionRequirement.AntibodyTypeId] + " AND " + TABLE.AntibodyParameter.HigherLevelOverrideIndicator + " = 1").Length > 0)
						{
							AntigenNegativeReqExists = true;
						}
					}
				}
			}

			if (PositiveABS == true || AntigenNegativeReqExists == true)
			{
				return false;
			}
			else
			{
				return true;
			}
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/29/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2802"> 
		///		<ExpectedInput>Guid</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="2811"> 
		///		<ExpectedInput>Invalid Guid</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Gets all orderedtests based on specimen GUID.
		/// </summary>
		/// <param name="patientSpecimenGuid"></param>
		/// <returns></returns>
		public static DataTable GetOrderedTests(Guid patientSpecimenGuid)
		{
			return DAL.OrderedTest.GetOrderedTests(patientSpecimenGuid);
		}


		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/24/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2812"> 
		///		<ExpectedInput>Valid patient specimen unique identifier</ExpectedInput>
		///		<ExpectedOutput>Data table of ordered test and ordered component results</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2813"> 
		///		<ExpectedInput>Invalid patient specimen unique identifier</ExpectedInput>
		///		<ExpectedOutput>Argument exception thrown</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Get Test Orders including non-finalized TRWs
		/// Implements BR_94.13
		/// </summary>
		/// <param name="patientSpecimenGuid"></param>
		/// <returns></returns>
		public static DataTable GetTestOrders(Guid patientSpecimenGuid)
		{
			if (patientSpecimenGuid == Guid.Empty)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("patient specimen unique identifier").ResString);
			}
			//
			return DAL.OrderedTest.GetTestOrders(patientSpecimenGuid);
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/24/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4475"> 
		///		<ExpectedInput>Valid ordered test unique identifier</ExpectedInput>
		///		<ExpectedOutput>Array list of specimen test result objects</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4476"> 
		///		<ExpectedInput>Invalid ordered test unique identifier</ExpectedInput>
		///		<ExpectedOutput>Argument exception thrown</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Static function returns the specimen test results for a given ordered test
		/// </summary>
		/// <param name="orderedTestGuid">ordered test unique idnetifier</param>
		/// <returns>Specimen test array list</returns>
		public static ArrayList GetSpecimenTestResultsForOrderedTest(Guid orderedTestGuid)
		{
			if (orderedTestGuid == Guid.Empty)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("ordered test unique identifier").ResString);
			}
			//
			System.Data.DataTable dt = DAL.OrderedTest.GetSpecimenTestResultsForOrderedTest(orderedTestGuid);
			//
			System.Collections.ArrayList returnList = new System.Collections.ArrayList();
			foreach(DataRow dr in dt.Rows)
			{
				BOL.SpecimenTest test = new BOL.SpecimenTest(dr);
				returnList.Add(test);
			}
			//
			return returnList;
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/24/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4477"> 
		///		<ExpectedInput>Valid ordered test unique identifier and reportable test id</ExpectedInput>
		///		<ExpectedOutput>Array list of subset of entire TAS specimen test result objects</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4478"> 
		///		<ExpectedInput>Invalid ordered test unique identifier</ExpectedInput>
		///		<ExpectedOutput>Argument exception thrown</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// GetSpecimenTestResultsForOrderedTest
		/// </summary>
		/// <param name="orderedTestGuid"></param>
		/// <param name="reportableTestID"></param>
		/// <returns></returns>
		public static ArrayList GetSpecimenTestResultsForOrderedTest(Guid orderedTestGuid, int reportableTestID)
		{
			if (orderedTestGuid == Guid.Empty)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("ordered test unique identifier").ResString);
			}
			//
			System.Data.DataTable dt = DAL.OrderedTest.GetSpecimenTestResultsForOrderedTest(orderedTestGuid,reportableTestID);
			//
			System.Collections.ArrayList returnList = new System.Collections.ArrayList();
			foreach(DataRow dr in dt.Rows)
			{
				BOL.SpecimenTest test = new BOL.SpecimenTest(dr);
				returnList.Add(test);
			}
			//
			return returnList;
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/24/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4471"> 
		///		<ExpectedInput>Valid ordered test and blood unit unique identifiers</ExpectedInput>
		///		<ExpectedOutput>Array list of completed blood unit test results</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4472"> 
		///		<ExpectedInput>Invalid ordered test and blood unit unique identifiers</ExpectedInput>
		///		<ExpectedOutput>Argument exception thrown</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Static function to retrieve a list of blood unit test results for a given ordered test (XMs only)
		/// </summary>
		/// <param name="orderedTestGuid">Ordered test unique identifier</param>
		/// <param name="bloodUnitGuid">Blood unit unique identifier</param>
		/// <returns>Array list of blood unit test oibjects</returns>
		public static ArrayList GetBloodUnitTestResultsForOrderedTest(Guid orderedTestGuid, Guid bloodUnitGuid)
		{
			if (orderedTestGuid == Guid.Empty)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("ordered test unique identifier").ResString);
			}
			//
			if (bloodUnitGuid == Guid.Empty)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("blood unit unique identifier").ResString);
			}
			//
			System.Data.DataTable dt = DAL.OrderedTest.GetBloodUnitTestResultsForOrderedTest(orderedTestGuid, bloodUnitGuid);
			//
			System.Collections.ArrayList returnList = new System.Collections.ArrayList();
			foreach(DataRow dr in dt.Rows)
			{
				BOL.BloodUnitTest test = new BOL.BloodUnitTest(dr);
				returnList.Add(test);
			}
			//
			return returnList;
		}

		///<Developers>
		///	<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/4/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="4283"> 
		///		<ExpectedInput>divisioncode, int, Date</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4284"> 
		///		<ExpectedInput>Invalid parameters</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// GetRequiredOrders
		/// </summary>
		/// <param name="divisionCode"></param>
		/// <param name="hoursNeeded"></param>
		/// <param name="lastChecked"></param>
		/// <returns>DataTable</returns>
		public static DataTable GetRequiredOrders(string divisionCode, int hoursNeeded, System.DateTime lastChecked)
		{
			return DAL.OrderedTest.GetRequiredOrders(divisionCode, hoursNeeded, lastChecked);
		}

		///<Developers>
		///	<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/4/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="4285"> 
		///		<ExpectedInput>DivisionCode, DateTime</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4286"> 
		///		<ExpectedInput>Invalid DivisionCode, DateTime</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// GetOrdersWithSpecimenInBloodBank
		/// </summary>
		/// <param name="divisionCode"></param>
		/// <param name="lastChecked"></param>
		/// <returns>DataTable</returns>
		public static DataTable GetOrdersWithSpecimenInBloodBank(string divisionCode, System.DateTime lastChecked)
		{	
			return DAL.OrderedTest.GetOrdersWithSpecimenInBloodBank(divisionCode, lastChecked);
		}

		///<Developers>
		///	<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/4/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="4287"> 
		///		<ExpectedInput>Division, Date</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4288"> 
		///		<ExpectedInput>Invalid Division, Date</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// GetNewOrders
		/// </summary>
		/// <param name="divisionCode"></param>
		/// <param name="lastChecked"></param>
		/// <returns>DataTable</returns>
		public static DataTable GetNewOrders(string divisionCode, System.DateTime lastChecked)
		{	
			return DAL.OrderedTest.GetNewOrders(divisionCode, lastChecked);
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/12/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="2814"> 
		///		<ExpectedInput>Date</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2815"> 
		///		<ExpectedInput>Invalid Date</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Static function, returns a data table containing recently added ordered orders
		/// that the user should be alerted about
		/// Implements BR_84.01, BR_84.02
		/// </summary>
		/// <param name="lastChecked">Date and time the user last checked for tests</param>
		/// <returns>Data table of ordered test information</returns>
		public static DataTable GetOrderAlerts(System.DateTime lastChecked)
		{
			DataTable orderAlerts;
			BOL.Division division = new BOL.Division(Common.LogonUser.LogonUserDivisionCode);
			//
			if (division.DivisionAlerts.IconOrderAlertType == Common.OrderAlertTypeCode.All_orders)
			{
				orderAlerts = BOL.OrderedTest.GetNewOrders(Common.LogonUser.LogonUserDivisionCode, lastChecked);
			}
			else if (division.DivisionAlerts.IconOrderAlertType == Common.OrderAlertTypeCode.Hour_limit)
			{
				orderAlerts = BOL.OrderedTest.GetRequiredOrders(Common.LogonUser.LogonUserDivisionCode, division.DivisionAlerts.IconOrdersNeededHours, lastChecked);
			}
			else
			{
				orderAlerts = BOL.OrderedTest.GetOrdersWithSpecimenInBloodBank(Common.LogonUser.LogonUserDivisionCode, lastChecked);
			}
			return orderAlerts;
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/12/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="2816"> 
		///		<ExpectedInput>Guid</ExpectedInput>
		///		<ExpectedOutput>DataSet</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2817"> 
		///		<ExpectedInput>Invalid Guid</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// GetRecentTestOrders
		/// </summary>
		/// <param name="patientGuid"></param>
		/// <returns></returns>
		public static DataSet GetRecentTestOrders(System.Guid patientGuid)
		{
			if (patientGuid == Guid.Empty)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("patient unique identifier").ResString);
			}
			return DAL.OrderedTest.GetRecentTestOrders(patientGuid);
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/9/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8189"> 
		///		<ExpectedInput>Valid patient guid</ExpectedInput>
		///		<ExpectedOutput>Count of the recent test orders</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="8190"> 
		///		<ExpectedInput>Invalid patient Guid</ExpectedInput>
		///		<ExpectedOutput>ArgumentException thrown</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Get a count of the recent test orders for a patient
		/// </summary>
		/// <param name="patientGuid">Patient Guid</param>
		/// <returns>Count of the recent test orders</returns>
		public static int GetRecentTestOrdersCount(System.Guid patientGuid)
		{
			if (patientGuid == Guid.Empty)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("patient unique identifier").ResString);
			}
			//
			return DAL.OrderedTest.GetRecentTestOrdersCount(patientGuid);
		}

		///<Developers>
		///	<Developer></Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/11/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7518"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7519"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Get the number of issued or transfused units for ordered test
		/// Implements BR_16.06
		/// </summary>
		/// <param name="orderedTestGuid"></param>
		/// <returns></returns>
		public static DataTable GetEmergencyIssuedUnitsForOrderedTest(System.Guid orderedTestGuid)
		{
			return DAL.OrderedTest.GetEmergencyIssuedUnitsForOrderedTest(orderedTestGuid);
		}

		///<Developers>
		///	<Developer></Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/11/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7520"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7521"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Get ordered test by Guid
		/// </summary>
		/// <param name="orderedTestGuid"></param>
		/// <returns></returns>
		public static DataTable GetOrderedTest(Guid orderedTestGuid)
		{
			return DAL.OrderedTest.GetOrderedTest(orderedTestGuid);
		}

		///<Developers>
		///	<Developer></Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/11/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7522"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7523"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
        /// BR_56.16, BR_56.19, BR_115.23, BR_115.27, BR_115.26
		/// </summary>
		/// <param name="testAbo"></param>
		/// <param name="testRh"></param>
		/// <param name="PreviousABOInterp"></param>
		/// <param name="PreviousRhInterp"></param>
		/// <param name="dtTestResults"></param>
		/// <returns></returns>
		public bool CheckForRepeatAboRhDiscrepancy(string testAbo, string testRh, ref string PreviousABOInterp, ref string PreviousRhInterp, ref DataTable dtTestResults)
		{
			string ABOInterpID = string.Empty;
			string RhInterpID = string.Empty;
			string ABOInterpIDTAS = string.Empty;
			string RhInterpIDTAS = string.Empty;

			DataTable dtBloodTestTypes = BOL.BloodTestType.GetBloodTestTypes(Convert.ToString((int)Common.OrderableTest.ABORh));
			DataTable dtBloodTestTypesTAS = BOL.BloodTestType.GetBloodTestTypes(Convert.ToString((int)Common.OrderableTest.TAS));
			dtTestResults = BOL.OrderedTest.GetTestResultsForTask(this.RelatedTestGuid);
								
			if (dtBloodTestTypes.Select("BloodTestName LIKE '*Interp*'").Length > 0 &&
				dtBloodTestTypesTAS.Select("BloodTestName LIKE '*Interp*'").Length > 0)
			{
				if (dtTestResults.Rows.Count > 0)
				{
					ABOInterpID = dtBloodTestTypes.Select("BloodTestName LIKE '*ABO Interp*'")[0][TABLE.BloodTestType.BloodTestTypeId].ToString();
					RhInterpID = dtBloodTestTypes.Select("BloodTestName LIKE '*Rh Interp*'")[0][TABLE.BloodTestType.BloodTestTypeId].ToString();
					ABOInterpIDTAS = dtBloodTestTypesTAS.Select("BloodTestName LIKE '*ABO Interp*'")[0][TABLE.BloodTestType.BloodTestTypeId].ToString();
					RhInterpIDTAS = dtBloodTestTypesTAS.Select("BloodTestName LIKE '*Rh Interp*'")[0][TABLE.BloodTestType.BloodTestTypeId].ToString();

					if (dtTestResults.Select("BloodTestTypeId = " + ABOInterpID).Length > 0 &&
						dtTestResults.Select("BloodTestTypeId = " + RhInterpID).Length > 0)
					{
						PreviousABOInterp = dtTestResults.Select("BloodTestTypeId = " + ABOInterpID)[0][TABLE.TestResult.TestResultId].ToString().Trim();
						PreviousRhInterp = dtTestResults.Select("BloodTestTypeId = " + RhInterpID)[0][TABLE.TestResult.TestResultId].ToString().Trim();
					}
					else if (dtTestResults.Select("BloodTestTypeId = " + ABOInterpIDTAS).Length > 0 &&
						dtTestResults.Select("BloodTestTypeId = " + RhInterpIDTAS).Length > 0)
					{
						PreviousABOInterp = dtTestResults.Select("BloodTestTypeId = " + ABOInterpIDTAS)[0][TABLE.TestResult.TestResultId].ToString().Trim();
						PreviousRhInterp = dtTestResults.Select("BloodTestTypeId = " + RhInterpIDTAS)[0][TABLE.TestResult.TestResultId].ToString().Trim();
					}
					else
					{
						PreviousABOInterp = string.Empty;
						PreviousRhInterp = string.Empty;
					}

					if (testAbo.Trim() != string.Empty && testRh.Trim() != string.Empty)
					{
						if (PreviousABOInterp != testAbo ||
							PreviousRhInterp != testRh)
						{
							return true;
						}
					}
				}
			}

			return false;
		}

		#endregion
	}
}
